/*
 * =============================================================================
 * Copyright (c) 2015-2016  Immersion Corporation.  All rights reserved.
 *                          Immersion Corporation Confidential and Proprietary
 *
 * MAY NOT BE USED OR DISTRIBUTED UNLESS EXPRESSLY LICENSED UNDER, AND
 * SUBJECT TO, A SEPARATE WRITTEN LICENSE AGREEMENT EXECUTED BETWEEN THE
 * APPLICABLE OEM/MANUFACTURER AND IMMERSION CORPORATION.
 *
 * =============================================================================
 */

package com.immersion.hapticmedia.sample;

import android.content.Context;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.Handler;
import android.view.SurfaceHolder;
import android.view.View;
import com.immersion.hapticmedia.AutoSyncPlayer;

import java.util.Stack;

public class MediaViewController implements SurfaceHolder.Callback,
        MediaPlayer.OnPreparedListener, MediaPlayer.OnCompletionListener {

    private static final String TAG = "MediaViewController";
    private Context mContext;
    private AutoSyncPlayer mMediaPlayer;

    enum State {
        IDLE,
        PREPARING,
        READY,
        PLAYING,
        PAUSED,
        SEEKING,
        STOPPED
    }

    private State mState;
    private MediaActivity mActivity;
    private Handler mHandler;
    private Stack<State> mStateStack;
    private boolean mIsHapticsMuted = false;
    private boolean mIsMediaRendered = false;
    private int mMediaDuration = 0;
    private Uri mVideoUri;

    private static final long SEEK_BAR_UPDATE_RATE_MS = 200;
    private static final long HIDE_MEDIA_CONTROLS_DELAY_MS = 4000;

    private Runnable mUpdateSeekBarRunnable = new Runnable() {
        @Override
        public void run() {
            int position = mMediaPlayer.getCurrentPosition();
            mActivity.updateSeekBar(position);
            mHandler.postDelayed(mUpdateSeekBarRunnable,
                    SEEK_BAR_UPDATE_RATE_MS);
        }
    };

    private Runnable mDelayedHideControls = new Runnable() {
        @Override
        public void run() {
            mActivity.hideMediaControls();
        }
    };

    private MediaPlayer.OnInfoListener mMediaInfoListener = new MediaPlayer.OnInfoListener() {
        @Override
        public boolean onInfo(MediaPlayer mp, int what, int extra) {
            switch (what) {
                case MediaPlayer.MEDIA_INFO_BUFFERING_START:
                    mediaPlayerBufferingStart();
                    break;
                case MediaPlayer.MEDIA_INFO_BUFFERING_END:
                    if (mIsMediaRendered)
                        mediaPlayerBufferingEnd();
                    break;
                case MediaPlayer.MEDIA_INFO_VIDEO_RENDERING_START:
                    mediaPlayerBufferingEnd();
                    mIsMediaRendered = true;
                    break;
                default:
            }
            return false;
        }
    };

    private MediaPlayer.OnBufferingUpdateListener mMediaBufferringListener =
            new MediaPlayer.OnBufferingUpdateListener() {
                @Override
                public void onBufferingUpdate(MediaPlayer mp, int percent) {
                    mediaPlayerBufferProgress(percent);
                }
            };

    private void mediaPlayerBufferProgress(int percent) {
        mActivity.updateSecondaryProgress((mMediaDuration/100)*percent);
    }

    public MediaViewController(Context context, MediaActivity activity) {
        mContext = context;
        mActivity = activity;
        mHandler = new Handler();
        mStateStack = new Stack<State>();
    }

    public void init(String username, String password, String dns) {
        mMediaPlayer = AutoSyncPlayer.create(mContext, username, password, dns);
        mMediaPlayer.setOnPreparedListener(this);
        if (mMediaPlayer.isHapticPlayerInitialized()) {
            mMediaPlayer.setOnInfoListener(mMediaInfoListener);
            mMediaPlayer.setOnCompletionListener(this);
            mMediaPlayer.setOnBufferingUpdateListener(mMediaBufferringListener);
            mMediaPlayer.setupHapticPlayerListeners();
        }
        mState = State.IDLE;
    }

    public void setMedia(Uri videoUri) {
        mVideoUri = videoUri;
    }

    public void openHaptics(String url) {
        if (mMediaPlayer.isHapticPlayerInitialized()) {
            mMediaPlayer.addHapticResource(url);
        }
    }


    @Override
    public void surfaceCreated(SurfaceHolder holder) {
        mMediaPlayer.setDisplay(holder);
    }

    @Override
    public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
    }

    @Override
    public void surfaceDestroyed(SurfaceHolder holder) {
    }

    @Override
    public void onPrepared(MediaPlayer mp) {
        mMediaDuration = mp.getDuration();
        mActivity.updateMaxSeekBar(mMediaDuration);
        resumePlayback();
    }

    @Override
    public void onCompletion(MediaPlayer mp) {
        stopUpdatingSeekBar();
        reset();
    }

    private void reset() {
        mActivity.updateSeekBar(0);
        mMediaPlayer.pause();
        mMediaPlayer.seekTo(0);
        mMediaPlayer.stop();
        mMediaPlayer.reset();
        mActivity.updatePlayPauseButton(false);
        mActivity.toggleSeekBar(false);

        mState = State.IDLE;
    }

    public void mute(View v) {
        if (mIsHapticsMuted) {
            mMediaPlayer.unmuteHaptics();
        } else {
            mMediaPlayer.muteHaptics();
        }

        mIsHapticsMuted = !mIsHapticsMuted;
        mActivity.updateMuteIcon(mIsHapticsMuted);
    }

    public void prepareMediaPlayer(MediaPlayer mp) {
        try {
            mp.prepareAsync();
        } catch (IllegalStateException e) {
            e.printStackTrace();
            mActivity.showUnableToPlayErrorDialog();
            return;
        }
        mState = State.PREPARING;
        mActivity.startLoadingProgressBar();
    }

    public void playPause(View v) {
        switch (mState) {
            // When controller is idle and button is clicked,
            // we start playback
            case IDLE:
                mMediaPlayer.openMedia(mContext, mVideoUri);
                prepareMediaPlayer(mMediaPlayer);
                break;
            case PREPARING:
                break;
            case READY:
                resumePlayback();
            // When controller state is playing and button is clicked,
            // we pause playback
            case PLAYING:
                pausePlayback();
                break;
            // When controller state is paused and button is clicked,
            // we resume playback
            case PAUSED:
                resumePlayback();
                break;
            default:
                break;
        }
    }

    public void stopPlayback() {
        mState = State.STOPPED;
        stopUpdatingSeekBar();
        mMediaPlayer.stop();
    }

    public void dispose() {
        mMediaPlayer.release();
        mHandler.removeCallbacks(null);
    }

    private void resumePlayback() {
        mState = State.PLAYING;
        mMediaPlayer.start();
        startUpdatingSeekBar();
        mActivity.updatePlayPauseButton(mState == State.PLAYING);
        mActivity.stopLoadingProgressBar();
        hideMediaControls();
    }

    private void unHideMediaControls() {
        mHandler.removeCallbacks(mDelayedHideControls);
    }

    private void hideMediaControls() {
        unHideMediaControls();
        mHandler.postDelayed(mDelayedHideControls,
                HIDE_MEDIA_CONTROLS_DELAY_MS);
    }

    public void pausePlayback() {
        switch (mState) {
            case PLAYING:
                mState = State.PAUSED;
                mMediaPlayer.pause();
                stopUpdatingSeekBar();
                mActivity.updatePlayPauseButton(mState == State.PLAYING);
                break;
            default:
                break;
        }
    }

    private void stopUpdatingSeekBar() {
        mHandler.removeCallbacks(mUpdateSeekBarRunnable);
    }

    private void startUpdatingSeekBar() {
        mHandler.post(mUpdateSeekBarRunnable);
    }

    public void startSeek() {
        unHideMediaControls();
        mStateStack.push(mState);
        mState = State.SEEKING;
    }

    public void endSeek() {
        mState = mStateStack.pop();
        hideMediaControls();
    }

    /**
     * When media player starts buffering, we stop
     * updating the seekbar
     */
    private void mediaPlayerBufferingStart() {
        stopUpdatingSeekBar();
        mActivity.startLoadingProgressBar();
    }

    /**
     * When media player is done buffering,
     *  we resume everything
     */
    private void mediaPlayerBufferingEnd() {
        startUpdatingSeekBar();
        mActivity.stopLoadingProgressBar();
    }


    public void seekTo(int position) {
        switch (mState) {
            case SEEKING:
                mMediaPlayer.seekTo(position);
                break;
        }
    }
}
