/*
 * =============================================================================
 * Copyright (c) 2015-2016  Immersion Corporation.  All rights reserved.
 *                          Immersion Corporation Confidential and Proprietary
 *
 * MAY NOT BE USED OR DISTRIBUTED UNLESS EXPRESSLY LICENSED UNDER, AND
 * SUBJECT TO, A SEPARATE WRITTEN LICENSE AGREEMENT EXECUTED BETWEEN THE
 * APPLICABLE OEM/MANUFACTURER AND IMMERSION CORPORATION.
 *
 * =============================================================================
 */

package com.immersion.hapticmedia.sample;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.net.Uri;
import android.os.Bundle;
import android.util.Log;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.view.WindowManager;
import android.view.animation.AccelerateInterpolator;
import android.view.animation.AlphaAnimation;
import android.view.animation.Animation;
import android.widget.ImageView;
import android.widget.ProgressBar;
import android.widget.SeekBar;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

public class MediaActivity extends Activity {

    private Context mApplicationContext;

    private MediaViewController mViewController;

    private String mResourceLocation = "android.resource://com.immersion.hapticmedia.sample/";
    private String mHaptFileName = "sample_haptics.hapt";

    private SeekBar mSeekBar;
    private ImageView mMuteButton;
    private ImageView mPlayPauseButton;
    private View mControlPanel;
    private SurfaceView mSurfaceView;
    private ProgressBar mMediaLoadingProgressBar;

    private Animation mFadeoutAnim;
    private boolean mIsMediaControlsHidden = false;

    // TODO: you need to pass in the credentials provided to you by Immersion.
    // It is recommended that you don't hard-code the credentials.
    // You should store them on the cloud, then fetch them, and pass them in at runtime.
    // Also, use the preferred DNS URL if it was provided to you by Immersion, otherwise use null.
    // Note that Haptics will not work with invalid credentials!
    private static final String USERNAME = null;
    private static final String PASSWORD = null;
    private static final String DNS      = null;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        // Render Activity's main view
        setContentView(R.layout.activity_main);

        // Store the application context
        mApplicationContext = getApplicationContext();

        // Initialize Media Controller
        mViewController = new MediaViewController(mApplicationContext, this);
        mViewController.init(USERNAME, PASSWORD, DNS);

        // Setup Media views
        setupMediaViews();

        // Create Uri object from resource id
        Uri videoUri = Uri.parse(mResourceLocation + R.raw.sample_file);

        mViewController.setMedia(videoUri);
        mViewController.openHaptics(getHapticUrl());

        // Make sure the screen doesn't dim while we're playing media
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();

        mViewController.stopPlayback();
        mViewController.dispose();
    }

    @Override
    protected void onPause() {
        super.onPause();

        mViewController.pausePlayback();
    }

    @Override
    protected void onResume() {
        super.onResume();

        unHideMediaControls();
    }

    private void setupMediaViews() {
        // Get the SurfaceView from layout
        mSurfaceView = (SurfaceView) findViewById(R.id.surfaceView);
        initSurfaceView(mViewController, mSurfaceView);
        SurfaceHolder surfaceHolder = mSurfaceView.getHolder();
        surfaceHolder.addCallback(mViewController);

        mSeekBar = (SeekBar)findViewById(R.id.seekBar);
        initSeekBar(mViewController, mSeekBar);

        mPlayPauseButton = (ImageView) findViewById(R.id.playPause);
        updatePlayPauseButton(false);
        initPlayPause(mViewController, mPlayPauseButton);

        mMuteButton = (ImageView)findViewById(R.id.mute);
        initMuteButton(mViewController, mMuteButton);

        mMediaLoadingProgressBar = (ProgressBar) findViewById(R.id.loading);
        mMediaLoadingProgressBar.setVisibility(View.GONE);

        mControlPanel = findViewById(R.id.media_controls_panel);

        mFadeoutAnim = new AlphaAnimation(1.0f, 0.0f);
    }

    public void startLoadingProgressBar() {
        mPlayPauseButton.setVisibility(View.GONE);
        mSeekBar.setEnabled(false);
        mMediaLoadingProgressBar.setVisibility(View.VISIBLE);
    }

    public void stopLoadingProgressBar() {
        mPlayPauseButton.setVisibility(View.VISIBLE);
        mSeekBar.setEnabled(true);
        mMediaLoadingProgressBar.setVisibility(View.GONE);
    }

    public void updateSeekBar(int position) {
        mSeekBar.setProgress(position);
    }

    public void updateSecondaryProgress(int position) {
        mSeekBar.setSecondaryProgress(position);
    }

    public void updateMuteIcon(boolean muted) {
        mMuteButton.setImageResource(
                muted ? R.drawable.tactile_icon_grey
                        : R.drawable.tactile_icon_white
        );
    }

    public void setUpAnimation(Animation animation) {
        AccelerateInterpolator interpolator = new AccelerateInterpolator();
        animation.reset();
        animation.setInterpolator(interpolator);
        animation.setStartTime(0);
        animation.setDuration(1000);
    }

    public void hideMediaControls() {
        setUpAnimation(mFadeoutAnim);
        mControlPanel.setAnimation(mFadeoutAnim);
        mControlPanel.setVisibility(View.GONE);
        mPlayPauseButton.setVisibility(View.GONE);
    }

    public void toggleMediaControls() {
        if (mIsMediaControlsHidden)
            unHideMediaControls();
        else
            hideMediaControls();

        mIsMediaControlsHidden = !mIsMediaControlsHidden;
    }

    private void unHideMediaControls() {
        mControlPanel.setVisibility(View.VISIBLE);
        if (!mMediaLoadingProgressBar.isShown())
            mPlayPauseButton.setVisibility(View.VISIBLE);
    }

    public void updatePlayPauseButton(boolean isPlaying) {
        mPlayPauseButton.setImageResource(
                isPlaying ? R.drawable.pause
                        : R.drawable.play
        );
    }

    public void toggleSeekBar(boolean enabled) {
        mSeekBar.setEnabled(enabled);
    }

    public void updateMaxSeekBar(int max) {
        mSeekBar.setMax(max);
    }

    public void showUnableToPlayErrorDialog() {
        AlertDialog alertDialog = new AlertDialog.Builder(this).create();
        alertDialog.setTitle("Error!");
        alertDialog.setMessage("Unable to prepare Media Player for playback!");
        alertDialog.setCancelable(true);
        alertDialog.show();
    }

    void initSeekBar(final MediaViewController mediaViewController,
                     SeekBar seekBar) {
        seekBar.setEnabled(false);
        seekBar.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
                mediaViewController.seekTo(progress);
            }

            @Override
            public void onStartTrackingTouch(SeekBar seekBar) {
                mediaViewController.startSeek();
            }

            @Override
            public void onStopTrackingTouch(SeekBar seekBar) {
                mediaViewController.endSeek();
            }
        });
    }

    void initPlayPause(final MediaViewController mediaViewController,
                       ImageView playPauseButton) {

        playPauseButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mediaViewController.playPause(v);
            }
        });
    }

    void initMuteButton(final MediaViewController mediaViewController,
                        ImageView muteButton) {
        muteButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mediaViewController.mute(v);
            }
        });
    }

    void initSurfaceView(final MediaViewController mediaViewController,
                         SurfaceView  surfaceView) {
        surfaceView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                toggleMediaControls();
            }
        });
    }

    public String getHapticUrl() {
        FileOutputStream fos = null;
        int maxLen = 1024, curLen = 0;
        byte[] tempBuffer = new byte[maxLen];
        InputStream is = getResources().openRawResource(R.raw.sample_haptics);
        try {
            fos = openFileOutput(mHaptFileName, Context.MODE_PRIVATE);
            curLen = is.available();
            while (curLen > 0)
            {
                curLen = is.read(tempBuffer);
                fos.write(tempBuffer, 0, curLen);
                curLen = is.available();
            }
            fos.close();
            is.close();
        } catch (FileNotFoundException fnfe) {
        } catch (IOException ioe) {
        } finally {
            try {
                fos.close();
                is.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        // Retrieve the path of the hapt file now on disk
        File file = getFilesDir();
        String path = file.getAbsolutePath() + File.separator + mHaptFileName;
        return path;
    }

}
