/*
 * =============================================================================
 * Copyright (c) 2016       Immersion Corporation.  All rights reserved.
 *                          Immersion Corporation Confidential and Proprietary
 *
 * MAY NOT BE USED OR DISTRIBUTED UNLESS EXPRESSLY LICENSED UNDER, AND
 * SUBJECT TO, A SEPARATE WRITTEN LICENSE AGREEMENT EXECUTED BETWEEN THE
 * APPLICABLE OEM/MANUFACTURER AND IMMERSION CORPORATION.
 *
 * =============================================================================
 */

package com.immersion.hapticmedia;

import android.content.Context;
import android.media.MediaPlayer;
import android.net.Uri;
import android.os.HandlerThread;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.util.Log;
import android.widget.Toast;

import java.io.IOException;
import java.lang.UnsatisfiedLinkError;

import com.immersion.touchsensesdk.HapticMediaPlayer;

/**
 * A media player that automatically synchronizes video/audio and haptics
 */
public class AutoSyncPlayer extends MediaPlayer {
    private static final String TAG = "AutoSyncPlayer";
    private HapticMediaPlayer mHapticPlayer;
    private HandlerThread mUpdateThread;

    private Handler mUpdateHandler;

    private static final int UPDATE_INTERVAL_MS = 1000;
    private static final int MSG_UPDATE = 0;

    private OnInfoListener mInfoListener;
    private OnBufferingUpdateListener mBufferingUpdateListener;
    private OnCompletionListener mCompletionListener;
    private OnSeekCompleteListener mSeekCompleteListener;
    private boolean mRenderingStarted = false;

    private int mResourceId;
    private int mEffectId;

    private Toast mErrorToast = null;

    private OnInfoListener mInternalInfoListener = new OnInfoListener() {
        @Override
        public boolean onInfo(MediaPlayer mp, int what, int extra) {
            switch (what) {
                case MediaPlayer.MEDIA_INFO_BUFFERING_START:
                    pauseHapticPlayer();
                    break;
                case MediaPlayer.MEDIA_INFO_BUFFERING_END:
                    mHapticPlayer.seek(mEffectId, mp.getCurrentPosition());
                    if (mRenderingStarted) {
                        startHapticPlayer();
                    }
                    break;
                case MediaPlayer.MEDIA_INFO_VIDEO_RENDERING_START:
                    mHapticPlayer.seek(mEffectId, mp.getCurrentPosition());
                    startHapticPlayer();
                    mRenderingStarted = true;
                    break;
                default:
            }

            return mInfoListener != null && mInfoListener.onInfo(
                    AutoSyncPlayer.this,
                    what, extra);
        }
    };

    private OnBufferingUpdateListener mInternalBufferUpdateListener = new OnBufferingUpdateListener() {
        @Override
        public void onBufferingUpdate(MediaPlayer mp, int percent) {
            if (mBufferingUpdateListener != null)
                mBufferingUpdateListener.onBufferingUpdate(
                        AutoSyncPlayer.this, percent);
        }
    };

    private OnCompletionListener mInternalCompletionListener = new OnCompletionListener() {
        @Override
        public void onCompletion(MediaPlayer mp) {
            stopHaptics();
            mRenderingStarted = false;

            if (mCompletionListener != null)
                mCompletionListener.onCompletion(AutoSyncPlayer.this);
        }
    };

    private OnSeekCompleteListener mInternalSeekListener = new OnSeekCompleteListener() {
        @Override
        public void onSeekComplete(MediaPlayer mp) {
            mHapticPlayer.seek(mEffectId, mp.getCurrentPosition());

            if (mSeekCompleteListener != null)
                mSeekCompleteListener.onSeekComplete(AutoSyncPlayer.this);
        }
    };

    private static class HandlerCallback implements Handler.Callback {
        private AutoSyncPlayer mMediaPlayer;
        private HapticMediaPlayer mHapticPlayer;

        HandlerCallback(AutoSyncPlayer mediaPlayer, HapticMediaPlayer
                hapticPlayer) {
            mMediaPlayer = mediaPlayer;
            mHapticPlayer = hapticPlayer;
        }

        @Override
        public boolean handleMessage(Message msg) {
            switch (msg.what) {
                case MSG_UPDATE:
                    int currentPos = mMediaPlayer.getCurrentPosition();
                    this.mHapticPlayer.update(mMediaPlayer.mEffectId, currentPos);
                    mMediaPlayer.mUpdateHandler.sendEmptyMessageDelayed(
                            MSG_UPDATE, UPDATE_INTERVAL_MS);
                    break;
                default:
            }
            return false;
        }

    }

    AutoSyncPlayer() {
        super();
    }

    /**
     * Creates a new autosync player object
     *
     * @param context Current context
     * @return an {@link AutoSyncPlayer} object
     */
    public static AutoSyncPlayer create(
            Context context, String username, String password, String dns) {

        AutoSyncPlayer autoSyncPlayer = new AutoSyncPlayer();

        HapticMediaPlayer hapticPlayer;
        try {
            hapticPlayer = HapticMediaPlayer.create(context, username, password, dns);
            int state = hapticPlayer.getPlayerInfo(HapticMediaPlayer.PlayerInfo.PLAYER_STATE);
            if (state != HapticMediaPlayer.PlayerState.INITIALIZED) {
                if (username == null || username.isEmpty() || password == null || password.isEmpty()) {
                    autoSyncPlayer.mErrorToast = Toast.makeText(context, "No credentials provided in MediaActivity.java!", Toast.LENGTH_LONG);
                    autoSyncPlayer.mErrorToast.show();
                } else if (state == HapticMediaPlayer.PlayerState.INVALID_CREDENTIALS) {
                    autoSyncPlayer.mErrorToast = Toast.makeText(context, "Credentials provided in MediaActivity.java are invalid!", Toast.LENGTH_LONG);
                    autoSyncPlayer.mErrorToast.show();
                }
                throw new Exception("Error creating HapticMediaPlayer. Error: " + state);
            }
        } catch (Exception|UnsatisfiedLinkError e) {
            Log.e(TAG, e.getMessage());
            return autoSyncPlayer;
        }

        HandlerThread thread = new HandlerThread("AutoSyncPlayerUpdateThread");
        thread.start();
        Looper looper = thread.getLooper();
        HandlerCallback callback = new HandlerCallback(autoSyncPlayer,
                hapticPlayer);
        Handler handler = new Handler(looper, callback);

        autoSyncPlayer.mUpdateHandler  = handler;
        autoSyncPlayer.mHapticPlayer   = hapticPlayer;
        autoSyncPlayer.mUpdateThread   = thread;

        return autoSyncPlayer;
    }

    public boolean isHapticPlayerInitialized() {
        return !(mHapticPlayer == null);
    }

    public void setupHapticPlayerListeners() {
        super.setOnInfoListener(mInternalInfoListener);
        super.setOnBufferingUpdateListener(mInternalBufferUpdateListener);
        super.setOnCompletionListener(mInternalCompletionListener);
        super.setOnSeekCompleteListener(mInternalSeekListener);
    }

    /**
     * Opens media Uri
     *
     * @param context Context
     * @param uri Uri for media
     */
    public void openMedia(Context context, Uri uri) {
        try {
            setDataSource(context, uri);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Opens haptic url
     *
     * @param url URL to haptic file
     */
    public void addHapticResource(String url) {
        mResourceId = mHapticPlayer.addResource(url,
                HapticMediaPlayer.HapticEffectType.SYNC_HAPTIC_EFFECT);
    }

    @Override
    public void setOnInfoListener(OnInfoListener listener) {
        mInfoListener = listener;
    }

    @Override
    public void setOnBufferingUpdateListener(OnBufferingUpdateListener listener) {
        mBufferingUpdateListener = listener;
    }

    @Override
    public void setOnCompletionListener(OnCompletionListener listener) {
        mCompletionListener = listener;
    }

    @Override
    public void setOnSeekCompleteListener(OnSeekCompleteListener listener) {
        mSeekCompleteListener = listener;
    }

    /**
     * Starts media and haptics playback
     *
     * @throws IllegalStateException
     */
    @Override
    public void start() throws IllegalStateException {
        super.start();

        if (mRenderingStarted) {
            startHapticPlayer();
        }

        if (mErrorToast != null) {
            mErrorToast.show();
        }
    }

    /**
     * Stops media and haptics playback
     *
     * @throws IllegalStateException
     */
    @Override
    public void stop() throws IllegalStateException {
        super.stop();
        stopHaptics();
    }

    private void stopHaptics() {
        if (isHapticPlayerInitialized()) {
            mHapticPlayer.stop(mEffectId);
            stopUpdateHapticPlayer();
        }
    }

    /**
     * Pauses media and haptics playback
     *
     * @throws IllegalStateException
     */
    @Override
    public void pause() throws IllegalStateException {
        super.pause();
        pauseHapticPlayer();
    }

    private void stopUpdateHapticPlayer() {
        mUpdateHandler.removeMessages(MSG_UPDATE);
    }

    private void startHapticPlayer() {
        int playerState = mHapticPlayer.getPlayerInfo(HapticMediaPlayer.PlayerInfo.PLAYER_STATE);
        int effectState = mHapticPlayer.getEffectInfo(mEffectId, HapticMediaPlayer.EffectInfo.EFFECT_STATE);

        if (playerState == HapticMediaPlayer.PlayerState.INITIALIZED) {
            if (effectState == HapticMediaPlayer.EffectState.PAUSED) {
                mHapticPlayer.resume(mEffectId);
                startUpdateHapticPlayer();
                return;
            } else if (effectState == HapticMediaPlayer.TouchSenseSDKError.INVALID_PARAMETER) {
                mEffectId = mHapticPlayer.play(mResourceId, HapticMediaPlayer.EffectPriority.NORMAL);
                startUpdateHapticPlayer();
                return;
            } else {
                Log.e(TAG, "Failed to start haptics. Invalid effect state " + effectState);
            }
        } else {
            Log.e(TAG, "Failed to start haptics. Invalid player state " + playerState);
        }
    }

    /**
     * Pauses haptic player if it's started
     */
    private void pauseHapticPlayer() {
        if (!isHapticPlayerInitialized()) {
            return;
        }

        int state = mHapticPlayer.getEffectInfo(mEffectId, HapticMediaPlayer.EffectInfo.EFFECT_STATE);
        if (state == HapticMediaPlayer.EffectState.PAUSED) {
            return;
        }

        if (state == HapticMediaPlayer.EffectState.BUFFERING ||
            state == HapticMediaPlayer.EffectState.PLAYING) {
            mHapticPlayer.pause(mEffectId);
            stopUpdateHapticPlayer();
            return;
        }

        stopUpdateHapticPlayer();
        Log.e(TAG, "Failed to pause haptics. Invalid effect state " + state);
    }

    private void startUpdateHapticPlayer() {
        mUpdateHandler.sendEmptyMessageDelayed(MSG_UPDATE, UPDATE_INTERVAL_MS);
    }

    /**
     * Seeks milliseconds into media
     *
     * @param ms Milliseconds since the beginning
     * @throws IllegalStateException
     */
    @Override
    public void seekTo(int ms) throws IllegalStateException {
        super.seekTo(ms);

        if (isHapticPlayerInitialized()) {
            if (mHapticPlayer.seek(mEffectId, ms) == HapticMediaPlayer.TouchSenseSDKError.INVALID_PARAMETER) {
                // mEffectId is not valid - possibly already deleted by seeking to end of the video
                // so we start playing at ms
                playAtPosition(ms);
            }
        }
    }

    private void playAtPosition(int ms) {
        mEffectId = mHapticPlayer.play(mResourceId, HapticMediaPlayer.EffectPriority.NORMAL);
        mHapticPlayer.seek(mEffectId, ms);
        startUpdateHapticPlayer();
    }

    /**
     * Releases HapticMediaPlayer
     */
    @Override
    public void release() {
        super.release();

        if (isHapticPlayerInitialized()) {
            stopUpdateHapticPlayer();
            mUpdateThread.quit();
            mHapticPlayer.dispose();
        }
    }

    /**
     * Mutes currently playing haptics
     */
    public void muteHaptics() {
        if (isHapticPlayerInitialized()) {
            mHapticPlayer.mute(mEffectId);
        }
    }

    /**
     * Unmutes currently playing haptics
     */
    public void unmuteHaptics() {
        if (isHapticPlayerInitialized()) {
            mHapticPlayer.unmute(mEffectId);
        }
    }
}
