﻿/*
** =======================================================================
** Copyright (c) 2016  Immersion Corporation.  All rights reserved.
**                     Immersion Corporation Confidential and Proprietary.
**
** =======================================================================
*/

using UnityEngine;
using System.IO;
using System.Collections.Generic;
using UnityEngine.UI;
using System;
using com.immersion.touchsensesdk;

public class Controller : MonoBehaviour {
    // TODO: you need to pass in the credentials provided to you by Immersion.
    // It is recommended that you don't hard-code the credentials.
    // You should store them on the cloud, then fetch them, and pass them in at runtime.
    // Also, use the preferred DNS URL if it was provided to you by Immersion, otherwise use null.
    // Note that Haptics will not work with invalid credentials!
    private string USERNAME = null;
    private string PASSWORD = null;
    private string DNS = null;

    private const string HAPT_RESOURCE_1 = "1.hapt";
    private const string HAPT_RESOURCE_2 = "2.hapt";
    private string[] mResourceNames;
    private List<int> mResourceIds;
    private Logger mLogger;

    private Button mButtonAddResources;
    private Button mButtonRemoveResources;
    private Button mButtonPlay1;
    private Button mButtonPlay2;
    private bool mResourcesAdded = false;

    void Start() {
        Init();
    }

    void Update() {
        if (Input.GetKeyDown(KeyCode.Escape)) {
            Application.Quit();
        }
    }

    private void Init() {
        mResourceNames = new string[] { HAPT_RESOURCE_1, HAPT_RESOURCE_2 };
        mResourceIds = new List<int>();
        mLogger = new Logger(GameObject.Find("Text-Log").GetComponent<Text>());
        InitButtons();
        InitHapticsManager();
    }

    private void InitHapticsManager() {
#if !UNITY_EDITOR
        HapticsManager.Init(USERNAME, PASSWORD, DNS);
        int playerState = HapticsManager.GetPlayerInfo(HapticMediaPlayer.PlayerInfo.PLAYER_STATE);
        if (playerState != (int)HapticMediaPlayer.PlayerState.INITIALIZED) {
            mLogger.PrintError("Haptic Media Player is not initialized. Error: " + HapticsManager.GetPlayerStateString(playerState));
        } else {
            mLogger.Print("Haptic Media Player ready!");
            HapticsManager.CopyStreamingAssetsToPersistentData(mResourceNames);
        }
#endif
    }

    private void InitButtons() {
        mButtonAddResources = GameObject.Find("Button-AddResources").GetComponent<Button>();
        mButtonRemoveResources = GameObject.Find("Button-RemoveResources").GetComponent<Button>();
        mButtonPlay1 = GameObject.Find("Button-Play1").GetComponent<Button>();
        mButtonPlay2 = GameObject.Find("Button-Play2").GetComponent<Button>();
    }

    private void ToggleButtons() {
        mButtonAddResources.interactable = !mResourcesAdded;
        mButtonRemoveResources.interactable = mResourcesAdded;
        mButtonPlay1.interactable = mResourcesAdded;
        mButtonPlay2.interactable = mResourcesAdded;
    }

    public void AddResources() {
        foreach (string resourceName in mResourceNames) {
            AddResource(resourceName);
        }

        if (mResourceIds.Count == mResourceNames.Length) {
            mResourcesAdded = true;
            ToggleButtons();
        }
    }

    private void AddResource(string haptName) {
#if !UNITY_EDITOR
        int res = HapticsManager.AddResource(HapticsManager.GetLocalResourceURI(haptName));
        if (res < 0) {
            mLogger.PrintError("Failed to add resource " + haptName + ". Error: " + HapticsManager.GetErrorString(res));
        } else {
            if (mResourceIds.Contains(res)) {
                mLogger.Print("Resource " + haptName + " already added. Resource ID: " + res);
            } else {
                mLogger.Print("Added resource " + haptName + ". Resource ID: " + res);
                mResourceIds.Add(res);
            }
        }
#endif
    }

    public void RemoveResources() {
        foreach (int resourceId in mResourceIds.ToArray()) {
            RemoveResource(resourceId);
            mResourceIds.Remove(resourceId);
        }

        if (mResourceIds.Count == 0) {
            mResourcesAdded = false;
            ToggleButtons();
        }
    }

    private void RemoveResource(int resourceId) {
#if !UNITY_EDITOR
        int res = HapticsManager.RemoveResource(resourceId);
        if (res < 0) {
            mLogger.PrintError("Failed to remove resource #" + resourceId + ". Error: " + HapticsManager.GetErrorString(res));
        } else {
            mLogger.Print("Removed resource #" + resourceId);
        }
#endif
    }

    public void PlayEffect(int num) {
#if !UNITY_EDITOR
        int resourceId = mResourceIds[num - 1];
        int res = HapticsManager.PlayResource(resourceId);
        if (res < 0) {
            mLogger.PrintError("Failed to play resource #" + num + ". Error: " + HapticsManager.GetErrorString(res));
        } else {
            mLogger.Print("Playing resource #" + resourceId + ". Effect ID: " + res);
        }
#endif
    }
}
