﻿/*
** ============================================================================
** Copyright (c) 2016-2017  Immersion Corporation.  All rights reserved.
**                          Immersion Corporation Confidential and Proprietary.
**
** ============================================================================
*/

using UnityEngine;
using System.IO;
using System;

namespace com.immersion.touchsensesdk {

    /// <summary>
    /// Singleton class to manage haptics playback.
    /// Can be invoked be calling HapticManager.MethodName(...)
    /// e.g. HapticsManager.Create(...)
    /// </summary>
    public class HapticsManager : MonoBehaviour {
        private static HapticsManager sInstance = null;
        private static HapticMediaPlayer sPlayer = null;

        private const int HAPTICS_MANAGER_NOT_READY = -100;
        private const string HAPTICS_MANAGER_NOT_READY_ERROR_MESSAGE = "Haptics Manager is not ready! Make sure you are " +
            "running on Android and that you called HapticsManager.Init(...) with the credentials provided by Immersion.";

    #region MonoBehaviour methods
        void Awake () {
            // Making sure there is only a single object created
            if (sInstance != null){
                Destroy(gameObject);
            } else {
                sInstance = this;
                GameObject.DontDestroyOnLoad(gameObject);
            }
        }

        void OnApplicationQuit() {
            if (sPlayer != null) {
                sPlayer.Dispose();
                sPlayer = null;
            }
        }
    #endregion

    #region Static utility functions
        /// <summary>
        /// Copies the files from the streaming assets zipped folder to the application's persistent data folder.
        /// Used for copying the .hapt files locally so their path is accessiable to the HapticMediaPlayer.
        /// </summary>
        /// <param name="assetsFilesNames">Assets files names.</param>
        public static void CopyStreamingAssetsToPersistentData (string[] assetsFilesNames) {
            string sourceDir = Application.streamingAssetsPath;
            string targetDir = Application.persistentDataPath;

            foreach (string fileName in assetsFilesNames) {
                using (WWW reader = new WWW (Path.Combine(sourceDir, fileName))) {
                    while (!reader.isDone) {}
                    File.WriteAllBytes (Path.Combine(targetDir, fileName), reader.bytes);
                }
            }
        }

        /// <summary> URI path to the local resource in the application's persistent data folder. </summary>
        /// <returns> The local resource URI </returns>
        /// <param name="haptFileName"> Hapt file name (e.g. effect_name.hapt). </param>
        public static string GetLocalResourceURI(string haptFileName) {
            return Path.Combine(Application.persistentDataPath, haptFileName);
        }

        /// <summary> Gets the <see cref="HapticMediaPlayer.TouchSenseSDKError"/> error string. </summary>
        /// <returns> The corresponsing <see cref="HapticMediaPlayer.TouchSenseSDKError"/> string. </returns>
        /// <param name="code"> <see cref="HapticMediaPlayer.TouchSenseSDKError"/> code. </param>
        public static string GetErrorString(int code) {
            return GetMessageString(typeof(HapticMediaPlayer.TouchSenseSDKError), code);
        }

        /// <summary> Gets the <see cref="HapticMediaPlayer.PlayerState"/> string. </summary>
        /// <returns> The corresponsing <see cref="HapticMediaPlayer.PlayerState"/> string. </returns>
        /// <param name="code" > <see cref="HapticMediaPlayer.PlayerState"/> code. </param>
        public static string GetPlayerStateString(int code) {

            return GetMessageString(typeof(HapticMediaPlayer.PlayerState), code);
        }

        /// <summary> Gets the <see cref="HapticMediaPlayer.EffectState"/> string. </summary>
        /// <returns> The corresponsing <see cref="HapticMediaPlayer.EffectState"/> string. </returns>
        /// <param name="code"> <see cref="HapticMediaPlayer.EffectState"/> code. </param>
        public static string GetEffectStateString(int code) {
            return GetMessageString(typeof(HapticMediaPlayer.EffectState), code);
        }

        private static string GetMessageString(Type enumType, int code) {
            if (code == HapticsManager.HAPTICS_MANAGER_NOT_READY) {
                return HAPTICS_MANAGER_NOT_READY_ERROR_MESSAGE;
            }
            return Enum.GetName(enumType, code);
        }
    #endregion

    #region Public API
        /// <summary>
        /// Creates a new instance of HapticMediaPlayer with the credentials provided by Immersion.
        /// For more information see <see cref="HapticMediaPlayer.Create"/>.
        /// </summary>
        public static void Init(string username, string password, string dnsURL = null) {
#if UNITY_ANDROID && !UNITY_EDITOR
            sPlayer = HapticMediaPlayer.Create(username, password, dnsURL);
#else
            sPlayer = null;
#endif
        }

        /// <summary>
        /// Adds a haptic resource if it doesn't exists and returns its unique ID.
        /// For more information see <see cref="HapticMediaPlayer.AddResource"/>.
        /// </summary>
        public static int AddResource(string haptUri, HapticMediaPlayer.HapticEffectType type = HapticMediaPlayer.HapticEffectType.ASYNC_HAPTIC_EFFECT) {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.AddResource(haptUri, (int)type);
        }

        /// <summary>
        /// Removes a haptic resource identified by its unique ID and stops any playing effects associated with it.
        /// For more information see <see cref="HapticMediaPlayer.RemoveResource"/>.
        /// </summary>
        public static int RemoveResource(int resourceId) {
                return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.RemoveResource(resourceId);
        }

        /// <summary>
        /// Plays a haptic resource identified by its unique ID and returns a unique effect ID.
        /// For more information see <see cref="HapticMediaPlayer.Play"/>.
        /// </summary>
        public static int PlayResource(int resourceId, HapticMediaPlayer.EffectPriority priority = HapticMediaPlayer.EffectPriority.NORMAL) {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.Play(resourceId, (int)priority);
        }

        /// <summary>
        /// Pauses an effect identified by its unique ID.
        /// For more information see <see cref="HapticMediaPlayer.Pause"/>.
        /// </summary>
        public static int PauseEffect(int effectId) {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.Pause(effectId);
        }

        /// <summary>
        /// Resumes an effect identified by its unique ID.
        /// For more information see <see cref="HapticMediaPlayer.Resume"/>.
        /// </summary>
        public static int ResumeEffect(int effectId) {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.Resume(effectId);
        }

        /// <summary>
        /// Stopd an effect identified by its unique ID.
        /// For more information see <see cref="HapticMediaPlayer.Stop"/>.
        /// </summary>
        public static int StopEffect(int effectId) {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.Stop(effectId);
        }

        /// <summary>
        /// Updates an effect identified by its unique ID to a specific timestamp.
        /// For more information see <see cref="HapticMediaPlayer.Update"/>.
        /// </summary>
        public static int UpdateEffect(int effectId, long timestampMS)
        {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.Update(effectId, timestampMS);
        }

        /// <summary>
        /// Seeks an effect identified by its unique ID to a specific timestamp.
        /// For more information see <see cref="HapticMediaPlayer.Seek"/>.
        /// </summary>
        public static int SeekEffect(int effectId, long timestampMS)
        {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.Seek(effectId, timestampMS);
        }

        /// <summary>
        /// Mutes an effect identified by its unique ID.
        /// For more information see <see cref="HapticMediaPlayer.Mute"/>.
        /// </summary>
        public static int MuteEffect(int effectId) {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.Mute(effectId);
        }

        /// <summary>
        /// Unmutes an effect identified by its unique ID.
        /// For more information see <see cref="HapticMediaPlayer.Unmute"/>.
        /// </summary>
        public static int UnmuteEffect(int effectId) {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.Unmute(effectId);
        }

        /// <summary>
        /// Retrieves information on the <see cref="HapticMediaPlayer"/> according to information code provided.
        /// For more information see <see cref="HapticMediaPlayer.GetPlayerInfo"/>.
        /// </summary>
        public static int GetPlayerInfo(HapticMediaPlayer.PlayerInfo info) {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.GetPlayerInfo((int)info);
        }

        /// <summary>
        /// Retrieves information on a resource or an effect according to its unique ID and information code provided.
        /// For more information see <see cref="HapticMediaPlayer.GetEffectInfo"/>.
        /// </summary>
        public static int GetEffectInfo(int effectId, HapticMediaPlayer.EffectInfo info) {
            return sPlayer == null ? HAPTICS_MANAGER_NOT_READY : sPlayer.GetEffectInfo(effectId, (int)info);
        }
    #endregion
    }
}
