﻿/*
** ============================================================================
** Copyright (c) 2016-2017  Immersion Corporation.  All rights reserved.
**                          Immersion Corporation Confidential and Proprietary.
**
** ============================================================================
*/

using UnityEngine;
using System.Runtime.InteropServices;
using System;

namespace com.immersion.touchsensesdk {

    public class HapticMediaPlayer {
    #region Native API definitions
        [DllImport("TouchSenseSDK")]
        private static extern IntPtr touchsensesdk_create(IntPtr jvm, ref IntPtr context, string username, string password, string dns);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_addResource(IntPtr player, string uri, int playbackStyle);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_removeResource(IntPtr player, int resourceId);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_play(IntPtr player, int resourceId, int priority);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_pause(IntPtr player, int effectId);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_resume(IntPtr player, int effectId);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_stop(IntPtr player, int effectId);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_update(IntPtr player, int effectId, long timestampMS);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_seek(IntPtr player, int effectId, long timestampMS);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_mute(IntPtr player, int effectId);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_unmute(IntPtr player, int effectId);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_getPlayerInfo(IntPtr player, int infoCodeNum);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_getEffectInfo(IntPtr player, int uid, int infoCodeNum);
        [DllImport("TouchSenseSDK")]
        private static extern int touchsensesdk_dispose(IntPtr player);
    #endregion

    #region Private Code
        private IntPtr mNativeInstance;
        private static AndroidJavaObject mContext;

        private HapticMediaPlayer(IntPtr nativeInstance) {
            mNativeInstance = nativeInstance;
        }
        private static IntPtr GetContextObject () {
            if (mContext == null) {
                using (AndroidJavaClass jclass = new AndroidJavaClass("com.unity3d.player.UnityPlayer")) {
                    mContext = jclass.GetStatic<AndroidJavaObject>("currentActivity");
                }
            }
            return mContext.GetRawObject();
        }
    #endregion

    #region Public API
        /// <summary> Creates a new instance of HapticMediaPlayer </summary>
        /// <param name="context"> An Android application Context object </param>
        /// <param name="username"> Mandatory username provided by Immersion </param>
        /// <param name="password"> Mandatory password provided by Immersion </param>
        /// <param name="dns"> Optional DNS URL provided by Immersion </param>
        /// <returns> HapticMediaPlayer instance </returns>
        public static HapticMediaPlayer Create(string username, string password, string dns = null) {
            IntPtr jcontext = GetContextObject();
            IntPtr nativeInstance = touchsensesdk_create(IntPtr.Zero, ref jcontext, username, password, dns);
            return new HapticMediaPlayer(nativeInstance);
        }

        /// <summary> Adds a haptic resource if it doesn't exists and returns its unique ID </summary>
        /// <param name="resourceURL"> URL to the effect .hapt file  </param>
        /// <param name="hapticEffectType"> one of the values in <see cref="HapticEffectType"/> </param>
        /// <returns> If successful, a positive value - the resource's unique ID associated with the provided URI <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           <see cref="TouchSenseSDKError.TOO_MANY_EFFECTS"/>"/> if exceeded the maximum number of resources <br>
        ///           <see cref="TouchSenseSDKError.OUT_OF_MEMORY"/> if there is insufficient memory to complete the operation <br>
        ///           <see cref="TouchSenseSDKError.INVALID_URI"/> if resourceURL is malformed or does not point to a .hapt <br>
        ///           <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if the input is invalid <br>
        ///           <see cref="TouchSenseSDKError.IO_ERROR"/> if uri refers to a local file and it cannot be read </returns>
        /// <remarks> The resource ID returned by this function is to be used when calling
        ///           <see cref="HapticMediaPlayer.RemoveResource"/>,
        ///           <see cref="HapticMediaPlayer.Play"/> and
        ///           <see cref="HapticMediaPlayer.GetEffectInfo"/> </remarks>
        public int AddResource(string uri, int playbackStyle) {
            return touchsensesdk_addResource(mNativeInstance, uri, playbackStyle);
        }

        /// <summary> Removes a haptic resource identified by its unique ID and stops any playing effects associated with it </summary>
        /// <param name="resourceID"> unique ID of the resource to be removed </param>
        /// <returns> <see cref="TouchSenseSDKError.SUCCESS"/> if the operation was successful <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if a resource with the provided ID is not found </returns>
        public int RemoveResource(int resourceId) {
            return touchsensesdk_removeResource(mNativeInstance, resourceId);
        }

        /// <summary> Plays a haptic resource identified by its unique ID and returns a unique effect ID </summary>
        /// <param name="resourceID"> the unique ID of the resource from which to create the effect </param>
        /// <param name="priority"> one of the <see cref="EffectPriority"/> values indicating the effect's priority </param>
        /// <returns> If successful, a positive value - the effect's unique ID associated with the resource <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if a resource with the provided ID is not found or priority is invalid <br>
        ///           <see cref="TouchSenseSDKError.INVALID_EFFECT"/> if the effect could not be created. See the log for more details <br>
        ///           <see cref="TouchSenseSDKError.HAPT_NOT_READY"/> if the resource is not in state READY <br>
        ///           <see cref="TouchSenseSDKError.TOO_MANY_CONCURRENT_EFFECTS"/> if maximum number of concurrent playbacks has been reached <br>
        ///           <see cref="TouchSenseSDKError.OUT_OF_MEMORY"/> if there is insufficient memory to complete the operation </returns>
        /// <remarks> The effect ID returned by this function is to be used when calling
        ///           <see cref="HapticMediaPlayer.Pause"/>,
        ///           <see cref="HapticMediaPlayer.Resume"/>,
        ///           <see cref="HapticMediaPlayer.Stop"/>,
        ///           <see cref="HapticMediaPlayer.Update"/>,
        ///           <see cref="HapticMediaPlayer.Seek"/>,
        ///           <see cref="HapticMediaPlayer.Mute"/>,
        ///           <see cref="HapticMediaPlayer.Unmute"/> and
        ///           <see cref="HapticMediaPlayer.GetEffectInfo"/></remarks>
        /// <remarks> The effect ID returned by this function is valid until the effect is stopped or completed </remarks>
        public int Play(int resourceId, int priority = (int) EffectPriority.NORMAL)
        {
            return touchsensesdk_play(mNativeInstance, resourceId, priority);
        }

        /// <summary> Pauses an effect identified by its unique ID </summary>
        /// <param name="effectID"> unique ID of the effect to be paused </param>
        /// <returns> <see cref="TouchSenseSDKError.SUCCESS"/> if the operation was successful <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if an effect with the provided ID is not found </returns>
        public int Pause(int effectId)
        {
            return touchsensesdk_pause(mNativeInstance, effectId);
        }

        /// <summary> Resumes an effect identified by its unique ID </summary>
        /// <param name="effectID"> unique ID of the effect to be resumed </param>
        /// <returns> <see cref="TouchSenseSDKError.SUCCESS"/> if the operation was successful <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if an effect with the provided ID is not found </returns>
        public int Resume(int effectId)
        {
            return touchsensesdk_resume(mNativeInstance, effectId);
        }

        /// <summary> Stops an effect identified by its unique ID </summary>
        /// <param name="effectID"> unique ID of the effect to be stopped </param>
        /// <returns> <see cref="TouchSenseSDKError.SUCCESS"/> if the operation was successful <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if an effect with the provided ID is not found </returns>
        public int Stop(int effectId)
        {
            return touchsensesdk_stop(mNativeInstance, effectId);
        }

        /// <summary> Updates an effect identified by its unique ID to a specific timestamp </summary>
        /// <param name="effectID"> unique ID of the effect to be updated </param>
        /// <param name="timestampMS"> timestamp (in milliseconds) to update to </param>
        /// <returns> <see cref="TouchSenseSDKError.SUCCESS"/> if the operation was successful <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if an effect with the provided ID is not found <br>
        ///           or if the given timestamp is invalid (negative or longer than the effect duration) </returns>
        /// <remarks> This function must be called at least once per second for <see cref="HapticEffectType.SYNC_HAPTIC_EFFECT"/> type effects </remarks>
        public int Update(int effectId, long timestampMS)
        {
            return touchsensesdk_update(mNativeInstance, effectId, timestampMS);
        }

        /// <summary> Seeks an effect identified by its unique ID to a specific timestamp </summary>
        /// <param name="effectID"> unique ID of the effect to be seeked </param>
        /// <param name="timestampMS"> timestamp (in milliseconds) to seek to </param>
        /// <returns> <see cref="TouchSenseSDKError.SUCCESS"/> if the operation was successful <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if an effect with the provided ID is not found <br>
        ///           or if the given timestamp is invalid (negative or longer than the effect duration) </returns>
        public int Seek(int effectId, long timestampMS)
        {
            return touchsensesdk_seek(mNativeInstance, effectId, timestampMS);
        }

        /// <summary> Mutes an effect identified by its unique ID </summary>
        /// <param name="effectID"> unique ID of the effect to be muted </param>
        /// <returns> <see cref="TouchSenseSDKError.SUCCESS"/> if the operation was successful <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if an effect with the provided ID is not found </returns>
        public int Mute(int effectId)
        {
            return touchsensesdk_mute(mNativeInstance, effectId);
        }

        /// <summary> Un-mutes an effect identified by its unique ID </summary>
        /// <param name="effectID"> unique ID of the effect to be un-muted </param>
        /// <returns> <see cref="TouchSenseSDKError.SUCCESS"/> if the operation was successful <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if an effect with the provided ID is not found </returns>
        public int Unmute(int effectId)
        {
            return touchsensesdk_unmute(mNativeInstance, effectId);
        }

        /// <summary> Retrieves information on the <see cref="HapticMediaPlayer"/> according to information code provided </summary>
        /// <param name="info"> one of the values in <see cref="PlayerInfo"/> </param>
        /// <returns> <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if info is not a valid value <br>
        ///           for <see cref="PlayerInfo.PLAYER_STATE"/>               - one of the values in <see cref="PlayerState"/> <br>
        ///           for <see cref="PlayerInfo.PLAYER_VERSION_MAJOR"/>       - Major version of the <see cref="HapticMediaPlayer"/> <br>
        ///           for <see cref="PlayerInfo.PLAYER_VERSION_MINOR"/>       - Minor version of the <see cref="HapticMediaPlayer"/> <br>
        ///           for <see cref="PlayerInfo.PLAYER_VERSION_BUILD"/>       - Build version of the <see cref="HapticMediaPlayer"/> <br>
        ///           for <see cref="PlayerInfo.PLAYER_VERSION_MAINTENANCE"/> - Maintenance version of the <see cref="HapticMediaPlayer"/> <br>
        ///           for <see cref="PlayerInfo.PLAYER_VERSION_PATCH"/>       - Patch version number of the <see cref="HapticMediaPlayer"/> <br>
        ///           if any of the PLAYER_VERSION_* values is missing <see cref="TouchSenseSDKError.LIB_VERSION_NOT_FOUND"/> <br>
        ///           for <see cref="PlayerInfo.PLAYER_PLAYBACK_TYPE"/>       - one of the values in <see cref="PlaybackType"/> <br>
        ///                                                                     or <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized </returns>
        public int GetPlayerInfo(int info)
        {
            return touchsensesdk_getPlayerInfo(mNativeInstance, info);
        }

        /// <summary> Retrieves information on a resource or an effect according to its unique ID and information code provided </summary>
        /// <param name="uid"> the unique ID of the resource or effect to retrieve information for </param>
        /// <param name="info"> one of the values in <see cref="EffectInfo"/> </param>
        /// <returns> <see cref="TouchSenseSDKError.INVALID_PARAMETER"/> if the provided uid is not found or info is not a valid value <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized <br>
        ///           for <see cref="EffectInfo.RESOURCE_STATE"/> - one of the values in <see cref="ResourceState"/> <br>
        ///           for <see cref="EffectInfo.EFFECT_STATE"/> - one of the values in <see cref="EffectState"/> <br>
        ///           for <see cref="EffectInfo.EFFECT_MUTED_STATE"/> - one of the values in <see cref="HapticEffectMutedState"/> <br>
        ///           for <see cref="EffectInfo.RESOURCE_TYPE"/> - one of the values in <see cref="HapticEffectType"/> <br>
        ///           for <see cref="EffectInfo.RESOURCE_LOCATION"/> - one of the values in <see cref="HapticEffectLocation"/> <br>
        ///           for <see cref="EffectInfo.RESOURCE_DURATION"/> - duration in milliseconds, or <see cref="TouchSenseSDKError.HAPT_NOT_READY"/> if the resource is no ready </returns>
        /// <remarks> Resource uid is retrieved when calling <see cref="HapticMediaPlayer.AddResource"/> </remarks>
        /// <remarks> Effect uid is retrieved when calling <see cref="HapticMediaPlayer.Play"/> </remarks>
        public int GetEffectInfo(int uid, int info)
        {
            return touchsensesdk_getEffectInfo(mNativeInstance, uid, info);
        }

        /// <summary> Disposes of the HapticMediaPlayer instance, all existing resources and any playing effects </summary>
        /// <returns> <see cref="TouchSenseSDKError.SUCCESS"/> if the operation succeeds <br>
        ///           <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> if the SDK is not initialized </returns>
        /// <remarks> This will place the <see cref="HapticMediaPlayer"/> in an uninitialized state and any future API calls will fail with <see cref="TouchSenseSDKError.PLAYER_NOT_INITIALIZED"/> </remarks>
        public int Dispose() {
            return touchsensesdk_dispose(mNativeInstance);
        }
    #endregion

    #region Public TouchSenseSDK Values
        /// <summary> Haptic Media Player error codes </summary>
        public enum TouchSenseSDKError {
            /// <summary> Call succeeded per API specification </summary>
            SUCCESS = 0,
            /// <summary> Call failed, invalid parameter passed </summary>
            INVALID_PARAMETER = -1,
            /// <summary> Call failed, invalid uri passed </summary>
            INVALID_URI = -2,
            /// <summary> Call failed, invalid effect </summary>
            INVALID_EFFECT = -3,
            /// <summary> Call failed, ran out of memory </summary>
            OUT_OF_MEMORY = -5,
            /// <summary> Call failed, error while reading/writing file </summary>
            IO_ERROR = -7,
            /// <summary> Call failed, hapt file is not ready for playback </summary>
            HAPT_NOT_READY = -9,
            /// <summary> Call failed, too many haptic resources were already loaded </summary>
            TOO_MANY_EFFECTS = -10,
            /// <summary> Call failed, player not initialized </summary>
            PLAYER_NOT_INITIALIZED = -11,
            /// <summary> Call failed, too many effects playing at the same time </summary>
            TOO_MANY_CONCURRENT_EFFECTS = -12,
            /// <summary> Call failed, invalid state </summary>
            INVALID_STATE = -13,
            /// <summary> Call failed, Haptic Media Player version not found </summary>
            LIB_VERSION_NOT_FOUND = -14,
            /// <summary> Call failed, TouchSense SDK suffered a fatal error and is now inoperative </summary>
            SDK_INOPERATIVE = -15,
        }

        /// <summary> Haptic Media player states </summary>
        public enum PlayerState {
            /// <summary> Initialization succeeded, HapticMediaPlayer is ready to use </summary>
            INITIALIZED = 0,
            /// <summary> Initialization failed, an incorrect parameter was specified </summary>
            BAD_PARAMETER = 1,
            /// <summary> Initialization failed, missing manifest permission declarations </summary>
            MISSING_PERMISSIONS = 2,
            /// <summary> Initialization failed, invalid credentials </summary>
            INVALID_CREDENTIALS = 3,
        }

        /// <summary> The haptic resource states </summary>
        public enum ResourceState {
            /// <summary> Haptic Resource not ready state </summary>
            NOT_READY = 10,
            /// <summary> Haptic Resource invalid state </summary>
            INVALID = 11,
            /// <summary> Haptic Resource ready state </summary>
            READY = 12,
        }

        /// <summary> The haptic effect states </summary>
        public enum EffectState {
            /// <summary> The initial state of a HapticEffect (temporary state) </summary>
            IDLE = 20,
            /// <summary> The Haptic Effect is currently playing </summary>
            PLAYING = 21,
            /// <summary> The Haptic Effect is currently paused </summary>
            PAUSED = 22,
            /// <summary> The Haptic Effect is paused due to buffering </summary>
            BUFFERING = 23,
            /// <summary> The Haptic Effect is paused due to a timeout </summary>
            TIMEOUT = 24,
        }

        /// <summary> Haptic Media Player effect types </summary>
        public enum HapticEffectType {
            /// <summary> Asynchronous haptic effect type </summary>
            ASYNC_HAPTIC_EFFECT = 1,
            /// <summary> Synchronizable haptic effect type </summary>
            SYNC_HAPTIC_EFFECT = 2,
        }

        /// <summary> Haptic Media Player effect locations </summary>
        public enum HapticEffectLocation {
            /// <summary> Local haptic effect </summary>
            LOCAL_HAPTIC_EFFECT = 1,
            /// <summary> Remote haptic effect </summary>
            REMOTE_HAPTIC_EFFECT = 2,
        }

        /// <summary> Haptic Media Player effect states </summary>
        public enum HapticEffectMutedState {
            /// <summary> Effect muted state </summary>
            MUTED = 30,
            /// <summary> Effect unmuted state </summary>
            UNMUTED = 31,
        }

        /// <summary> Types of information that can be retrieved from calling  <see cref="HapticMediaPlayer.GetPlayerInfo"/> </summary>
        public enum PlayerInfo {
            /// <summary> State of the HapticMediaPlayer </summary>
            PLAYER_STATE = 40,
            /// <summary> Major version of the HapticMediaPlayer </summary>
            PLAYER_VERSION_MAJOR = 41,
            /// <summary> Minor version of the HapticMediaPlayer </summary>
            PLAYER_VERSION_MINOR = 42,
            /// <summary> Build version of the HapticMediaPlayer </summary>
            PLAYER_VERSION_BUILD = 43,
            /// <summary> Maintenance version of the HapticMediaPlayer </summary>
            PLAYER_VERSION_MAINTENANCE = 44,
            /// <summary> Patch version number of the HapticMediaPlayer </summary>
            PLAYER_VERSION_PATCH = 45,
            /// <summary> Playback type of the HapticMediaPlayer </summary>
            PLAYER_PLAYBACK_TYPE = 46,
        }

        /// <summary> Types of information that can be retrieved from calling <see cref="HapticMediaPlayer.GetEffectInfo"/> </summary>
        public enum EffectInfo {
            /// <summary> State of a haptic resource </summary>
            RESOURCE_STATE = 50,
            /// <summary> State of a haptic effect </summary>
            EFFECT_STATE = 51,
            /// <summary> Muted state of a haptic effect </summary>
            EFFECT_MUTED_STATE = 52,
            /// <summary> Sync or async haptic resource </summary>
            RESOURCE_TYPE = 53,
            /// <summary> Local or remote haptic resource </summary>
            RESOURCE_LOCATION = 54,
            /// <summary> Length in milliseconds of haptic resource </summary>
            RESOURCE_DURATION = 55,
        }

        /// <summary> Effect priorities </summary>
        public enum EffectPriority {
            LOWEST = 0,
            LOW = 1,
            NORMAL = 2,
            HIGH = 3,
            HIGHEST = 4,
            NUM_OF_PRIORITIES = 5,
        }

        /// <summary> Types of playback </summary>
        public enum PlaybackType {
            /// <summary> No Playback </summary>
            NO_PLAYBACK = 0,
            /// <summary> Playback using Android Vibrate API </summary>
            ANDROID_VIBRATE_API_PLAYBACK = 1,
            /// <summary> Playback using TouchSense </summary>
            TOUCHSENSE_PLAYBACK = 2,
        }
    #endregion
    }
}
