/*
** =============================================================================
** Copyright (c) 2016  Immersion Corporation. All rights reserved.
**                     Immersion Corporation Confidential and Proprietary.
**
** File:
**     GIFMovieView.java
**
** Description:
**     Class responsible of animating sticker GIFs
**
** =============================================================================
*/
package com.immersion.stickersampleapp.views;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Movie;
import android.util.AttributeSet;
import android.view.View;
import java.io.InputStream;

public class GIFMovieView extends View {
    private static final String TAG = "GIFMovieView";
    private int mMeasuredMovieWidth = 0, mMeasuredMovieHeight = 0;
    private float mScale = 0f;
    private Movie mMovie;
    private long mMovieStart;
    private boolean mAnimationFinished = true;
    private boolean mAnimationPaused = false;
    private int mCurrentTime = 0;

    public GIFMovieView(Context context) {
        super(context);
    }

    public GIFMovieView(Context context, AttributeSet attrs) {
        super(context, attrs);
    }

    public GIFMovieView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
    }

    public boolean isAnimationFinished() {
        return mAnimationFinished;
    }

    public boolean isAnimationPaused() {
        return mAnimationPaused;
    }

    public int getCurrentPosition() {
        return mCurrentTime;
    }

    public void pauseAnimation() {
        mAnimationPaused = true;
    }

    public void resumeAnimation() {
        mAnimationPaused = false;
        mMovieStart = 0;
        invalidate();
    }

    public void setResource(InputStream is) {
        mAnimationFinished = false;

        if (mMovieStart != 0) {
            mMovieStart = 0;
            mCurrentTime = 0;
        }
        mMovie = Movie.decodeStream(is);

        invalidate();
        requestLayout();
    }

    public void resetPlayback() {
        mAnimationFinished = false;
        mMovieStart = 0;
        mCurrentTime = 0;
        invalidate();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {

        if (mMovie != null) {
            int movieWidth = mMovie.width();
            int movieHeight = mMovie.height();

            /*
            * Calculate horizontal scaling
            */
            float scaleH = 1f;
            int measureModeWidth = MeasureSpec.getMode(widthMeasureSpec);
            if (measureModeWidth != MeasureSpec.UNSPECIFIED) {
                int maximumWidth = MeasureSpec.getSize(widthMeasureSpec);
                if (movieWidth > maximumWidth) {
                    scaleH = (float) movieWidth / (float) maximumWidth;
                }
            }

            /*
            * calculate vertical scaling
            */
            float scaleW = 1f;
            int measureModeHeight = MeasureSpec.getMode(heightMeasureSpec);

            if (measureModeHeight != MeasureSpec.UNSPECIFIED) {
                int maximumHeight = MeasureSpec.getSize(heightMeasureSpec);
                if (movieHeight > maximumHeight) {
                    scaleW = (float) movieHeight / (float) maximumHeight;
                }
            }

            /*
            * calculate overall scale
            */
            mScale = 1f / Math.max(scaleH, scaleW);
            mMeasuredMovieWidth = (int) (movieWidth * mScale);
            mMeasuredMovieHeight = (int) (movieHeight * mScale);
            setMeasuredDimension(mMeasuredMovieWidth, mMeasuredMovieHeight);
        } else {
            setMeasuredDimension(0, 0);
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        final long now = android.os.SystemClock.uptimeMillis();
        if (!mAnimationPaused) {

            if (mMovieStart == 0) { // first time
                mMovieStart = now - mCurrentTime;
                mCurrentTime = 0;
            }

            if (mMovie != null) {
                int dur = mMovie.duration();

                if (dur == 0) {
                    dur = 1000;
                }
                final int relTime = (int) ((now - mMovieStart) % dur);
                mCurrentTime = relTime;

                if (mAnimationFinished) {
                    mMovie.setTime(mMovie.duration());
                } else {
                    mMovie.setTime(relTime);
                }

                canvas.save(Canvas.MATRIX_SAVE_FLAG);
                canvas.scale(mScale, mScale);
                mMovie.draw(canvas, 0, 0);
                canvas.restore();

                if (dur > (now - mMovieStart))
                    invalidate();
                else if (dur <= (now - mMovieStart))
                    mAnimationFinished = true;
            }
        } else {
            mCurrentTime = (int) ((now - mMovieStart) % mMovie.duration());
            mMovie.setTime(mCurrentTime);
            canvas.save(Canvas.MATRIX_SAVE_FLAG);
            canvas.scale(mScale, mScale);
            mMovie.draw(canvas, 0, 0);
            canvas.restore();
        }
    }
}