/*
** =============================================================================
** Copyright (c) 2016  Immersion Corporation. All rights reserved.
**                     Immersion Corporation Confidential and Proprietary.
**
** File:
**     MainActivity.java
**
** Description:
**     Main activity of the sticker sample app application
**
** =============================================================================
*/
package com.immersion.stickersampleapp;

import android.Manifest;
import android.app.Activity;
import android.content.pm.PackageManager;
import android.graphics.Typeface;
import android.os.Bundle;
import android.support.v4.app.ActivityCompat;
import android.support.v4.content.ContextCompat;
import android.util.Log;
import android.view.MotionEvent;
import android.view.View;
import android.view.Window;
import android.widget.Button;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.immersion.stickersampleapp.views.GIFMovieView;
import com.immersion.touchsensesdk.HapticMediaPlayer;

import java.io.InputStream;
import java.util.HashMap;

public class MainActivity extends Activity implements ActivityCompat.OnRequestPermissionsResultCallback {

    public static final String TAG = "Haptic Sticker Demo";
    private static final HashMap<Integer, Integer> STICKERS = new HashMap<Integer, Integer>();

    static {
        STICKERS.put(R.id.momo_byebye, R.drawable.momo_byebye);
        STICKERS.put(R.id.momo_camera, R.drawable.momo_camera);
        STICKERS.put(R.id.momo_heart, R.drawable.momo_heart);
        STICKERS.put(R.id.yoyo_angry, R.drawable.yoyo_angry);
        STICKERS.put(R.id.yoyo_kiss, R.drawable.yoyo_kiss);
        STICKERS.put(R.id.yoyo_nervous, R.drawable.yoyo_nervous);
        STICKERS.put(R.id.yoyo_sad, R.drawable.yoyo_sad);
        STICKERS.put(R.id.yoyo_sleepy, R.drawable.yoyo_sleepy);
        STICKERS.put(R.id.yoyo_smile, R.drawable.yoyo_smile);
    }

    private HapticManager mHapticManager;
    private GIFMovieView mStickerGIF = null;
    private InputStream mGIFStream = null;
    private Button mAboutButton;
    private Button mOkButton;
    private RelativeLayout mAboutBackgroundView;
    private RelativeLayout mAboutDialog;
    private TextView mAppTitleLabel;
    private TextView mAboutLabel;
    private TextView mAppVersionLabel;
    private TextView mSDKVersionLabel;

    private boolean mStickerEnabled = true;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        this.requestWindowFeature(Window.FEATURE_NO_TITLE);
        setContentView(R.layout.activity_main);

        mHapticManager = HapticManager.getInstance(this);
        mHapticManager.addEffects();

        mStickerGIF = new GIFMovieView(this);

        mAboutBackgroundView = (RelativeLayout) findViewById(R.id.about_view);
        mAboutDialog = (RelativeLayout) findViewById(R.id.about_dialog);
        mAboutButton = (Button) findViewById(R.id.btn_about);
        mOkButton = (Button) findViewById(R.id.ok_button);
        mSDKVersionLabel = (TextView) findViewById(R.id.sdk_version_label);
        mAppVersionLabel = (TextView) findViewById(R.id.app_version_label);
        mAppTitleLabel = (TextView) findViewById(R.id.title_label);
        mAboutLabel = (TextView) findViewById(R.id.about_text);

        Typeface robotoTypeface = Typeface.createFromAsset(getAssets(), "fonts/Roboto-Medium.ttf");

        mAppTitleLabel.setTypeface(robotoTypeface);
        mAboutLabel.setTypeface(robotoTypeface);

        ImageView heartSticker =(ImageView)findViewById(R.id.momo_heart);
        ImageView byeSticker =(ImageView)findViewById(R.id.momo_byebye);
        ImageView cameraSticker =(ImageView)findViewById(R.id.momo_camera);
        ImageView smileSticker =(ImageView)findViewById(R.id.yoyo_smile);
        ImageView angrySticker =(ImageView)findViewById(R.id.yoyo_angry);
        ImageView kissSticker =(ImageView)findViewById(R.id.yoyo_kiss);
        ImageView sadSticker =(ImageView)findViewById(R.id.yoyo_sad);
        ImageView sleepySticker =(ImageView)findViewById(R.id.yoyo_sleepy);
        ImageView nervousSticker =(ImageView)findViewById(R.id.yoyo_nervous);

        heartSticker.setOnClickListener(new onStickerListener("momo_heart"));
        byeSticker.setOnClickListener(new onStickerListener("momo_byebye"));
        cameraSticker.setOnClickListener(new onStickerListener("momo_camera"));
        smileSticker.setOnClickListener(new onStickerListener("yoyo_smile"));
        angrySticker.setOnClickListener(new onStickerListener("yoyo_angry"));
        kissSticker.setOnClickListener(new onStickerListener("yoyo_kiss"));
        sadSticker.setOnClickListener(new onStickerListener("yoyo_sad"));
        sleepySticker.setOnClickListener(new onStickerListener("yoyo_sleepy"));
        nervousSticker.setOnClickListener(new onStickerListener("yoyo_nervous"));

        mAboutBackgroundView.setBackgroundColor(getResources().getColor(R.color.black));
        mAboutBackgroundView.setAlpha(0.4f);
        mAboutDialog.bringToFront();

        try {
            String versionName = getPackageManager()
                    .getPackageInfo(getPackageName(), 0).versionName;
            mAppVersionLabel.setText(versionName);
        } catch (PackageManager.NameNotFoundException e) {
            mAppVersionLabel.setText("NOT FOUND");
        }

        mSDKVersionLabel.setText(mHapticManager.getSDKVersion());

        mAboutBackgroundView.setVisibility(View.INVISIBLE);
        mAboutDialog.setVisibility(View.INVISIBLE);

        mAboutButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mAboutBackgroundView.setVisibility(View.VISIBLE);
                mAboutDialog.setVisibility(View.VISIBLE);
                mOkButton.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        mAboutBackgroundView.setVisibility(View.INVISIBLE);
                        mAboutDialog.setVisibility(View.INVISIBLE);
                    }
                });
            }
        });
    }

    @Override
    public void onPause() {
        super.onPause();

        mHapticManager.stopAll();
    }

    @Override
    public void onResume() {
        super.onResume();

        if (mHapticManager == null) {
            mHapticManager = HapticManager.getInstance(this);
            mHapticManager.addEffects();
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();

        mHapticManager.dispose();
        mHapticManager = null;
    }

    /**
     * Callback for the result from requesting permissions. This method
     * is invoked for every call on {@link #requestPermissions(Activity,
     * String[], int)}.
     * <p>
     * <strong>Note:</strong> It is possible that the permissions request interaction
     * with the user is interrupted. In this case you will receive empty permissions
     * and results arrays which should be treated as a cancellation.
     * </p>
     *
     * @param requestCode  The request code passed in {@link #requestPermissions(
     *Activity, String[], int)}
     * @param permissions  The requested permissions. Never null.
     * @param grantResults The grant results for the corresponding permissions
     *                     which is either {@link PackageManager#PERMISSION_GRANTED}
     *                     or {@link PackageManager#PERMISSION_DENIED}. Never null.
     * @see #requestPermissions(Activity, String[], int)
     */
    @Override
    public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        if (requestCode == HapticManager.EXTERNAL_STORAGE_PERMISSION_REQUEST) {
            if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                mHapticManager.addEffects();
            } else {
                Log.e(TAG, "This app requires access to external storage to load haptic resources!");
            }
        }
    }

    private class onStickerListener implements View.OnClickListener {

        private String mEffectName = null;

        public onStickerListener(String effectName) {
            mEffectName = effectName;
        }

        @Override
        public void onClick(View v) {

            if (mStickerEnabled) {
                try {
                    mGIFStream = getResources().openRawResource(STICKERS.get(v.getId()));
                } catch (Exception e) {
                    Log.e(TAG, "Couldn't retrieve effects' stream!");
                    return;
                }

                mStickerGIF = (GIFMovieView) findViewById(R.id.stickerView);
                mStickerGIF.setResource(mGIFStream);

                mStickerGIF.setOnTouchListener(new View.OnTouchListener() {
                    @Override
                    public boolean onTouch(View v, MotionEvent event) {

                        if (mStickerEnabled) {
                            if (mStickerGIF.isAnimationFinished()) {
                                mStickerGIF.resetPlayback();
                                mHapticManager.playEffect(mEffectName, HapticMediaPlayer.EffectPriority.NORMAL);
                            } else {
                                switch (event.getAction()) {
                                    case MotionEvent.ACTION_DOWN:
                                        mStickerGIF.pauseAnimation();
                                        mHapticManager.pauseEffect(mEffectName);
                                        break;

                                    case MotionEvent.ACTION_UP:
                                        mStickerGIF.resumeAnimation();
                                        mHapticManager.seek(mEffectName,
                                                mStickerGIF.getCurrentPosition());
                                        mHapticManager.resumeEffect(mEffectName);
                                        break;
                                }
                            }
                        }
                        return true;
                    }
                });

                mHapticManager.playEffect(mEffectName, HapticMediaPlayer.EffectPriority.NORMAL);
            }
        }
    }
}
