/*
** =============================================================================
** Copyright (c) 2016-2018  Immersion Corporation. All rights reserved.
**                          Immersion Corporation Confidential and Proprietary.
**
** File:
**     HapticManager.java
**
** Description:
**     Class responsible for all operations on the HapticMediaPlayer instance.
**
** =============================================================================
*/
package com.immersion.stickersampleapp;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Environment;
import android.support.v4.app.ActivityCompat;
import android.support.v4.content.ContextCompat;
import android.util.Log;
import android.widget.Toast;

import com.immersion.touchsensesdk.HapticMediaPlayer;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.lang.UnsatisfiedLinkError;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

/**
 * A singleton class the manages the Haptic Player instance. Any call to
 * HapticMediaPlayer happens through this class.
 */
public class HapticManager {
    private final static String TAG = "HapticManager";
    public static final String HAPT_FILE_EXTENSION = ".hapt";
    private static HapticManager s_instance = null;

    //If effects are to be loaded from external storage, create a directory under the sdcard,
    //and provide its path in the variable EXTERNAL_STORAGE_DIRECTORY
    private final static String EXTERNAL_STORAGE_DIRECTORY = "/immersion/StickerEffects";

    //Permission request code to read from external if effects are loaded from there
    public static final int EXTERNAL_STORAGE_PERMISSION_REQUEST = 0;

    /**
     * A class used to store the mappings between the effect names and their ids
     */
    private class HapticEffect {
        private String mEffectName;
        private int mResourceId;
        private int mEffectId;

        public HapticEffect(String effectName, int resourceId) {
            mEffectName = effectName;
            mResourceId = resourceId;
        }
    }

    /**
     * A list used to store the effects being played inside HapticMediaPlayer.
     */
    private List<HapticEffect> mStickerEffects;

    /**
     * A map of the error values that can be received from the haptic media player
     */
    private HashMap<Integer, String> mErrorValues;

    /**
     * Haptic media player instance that is the entry point to TouchSenseSDK
     * library.
     */
    private HapticMediaPlayer mHapticMediaPlayer;
    private Activity mActivity;

    // TODO: you need to pass in the credentials provided to you by Immersion.
    // It is recommended that you don't hard-code the credentials.
    // You should store them on the cloud, then fetch them, and pass them in at runtime.
    // Also, use the preferred DNS URL if it was provided to you by Immersion, otherwise use null.
    // Note that Haptics will not work with invalid credentials!
    private static final String USERNAME = null;
    private static final String PASSWORD = null;
    private static final String DNS      = null;

    private Toast mErrorToast = null;
    private HapticManager(Activity activity) {
        mActivity = activity;
        mStickerEffects = new ArrayList<HapticEffect>();

        initHapticErrorValues();

        try {
            mHapticMediaPlayer = HapticMediaPlayer.create(activity, USERNAME, PASSWORD, DNS);
            int state = mHapticMediaPlayer.getPlayerInfo(HapticMediaPlayer.PlayerInfo.PLAYER_STATE);
            if (state != HapticMediaPlayer.PlayerState.INITIALIZED) {
                if (USERNAME == null || USERNAME.isEmpty() || PASSWORD == null || PASSWORD.isEmpty()) {
                    mErrorToast = Toast.makeText(activity, "No credentials provided in HapticManager.java!", Toast.LENGTH_LONG);
                    mErrorToast.show();
                } else if (state == HapticMediaPlayer.PlayerState.INVALID_CREDENTIALS) {
                    mErrorToast = Toast.makeText(activity, "Credentials provided in HapticManager.java are invalid!", Toast.LENGTH_LONG);
                    mErrorToast.show();
                }
                throw new Exception("Error creating HapticMediaPlayer. Error: " + mErrorValues.get(state));
            }
        } catch (Exception|UnsatisfiedLinkError e) {
            Log.e(TAG, e.getMessage());
        }
    }

    public synchronized static HapticManager getInstance(Activity activity) {

        if (s_instance == null) {
            s_instance = new HapticManager(activity);
        }
        return s_instance;
    }

    public boolean isHapticsInitialized() {
        return mHapticMediaPlayer != null;
    }

    public boolean isEffectPlaying(int effectId) {
        return isHapticsInitialized() &&
                (mHapticMediaPlayer.getEffectInfo(effectId,
                        HapticMediaPlayer.EffectInfo.EFFECT_STATE) ==
                        HapticMediaPlayer.EffectState.PLAYING);
    }
    /**
     * Method to retrieve the version of the TouchSenseSDK library
     *
     * @return version of TouchSenseSDK library
     */
    public String getSDKVersion() {
        if (!isHapticsInitialized()) {
            return "NOT FOUND";
        }
        int[] versionInfoCodeNumbers = {
                HapticMediaPlayer.PlayerInfo.PLAYER_VERSION_MAJOR,
                HapticMediaPlayer.PlayerInfo.PLAYER_VERSION_MINOR,
                HapticMediaPlayer.PlayerInfo.PLAYER_VERSION_BUILD,
                HapticMediaPlayer.PlayerInfo.PLAYER_VERSION_MAINTENANCE,
                HapticMediaPlayer.PlayerInfo.PLAYER_VERSION_PATCH
        };

        StringBuilder str = new StringBuilder();
        for (int i = 0; i < versionInfoCodeNumbers.length; i++) {
            int versionNumPart = mHapticMediaPlayer.getPlayerInfo(versionInfoCodeNumbers[i]);

            if (versionNumPart == HapticMediaPlayer.TouchSenseSDKError.LIB_VERSION_NOT_FOUND) {
                return "UNKNOWN";
            }
            if (i != 0) {
                str.append(".");
            }
            str.append(versionNumPart);
        }

        // Omitting trailing ".0"s
        return str.toString().replaceAll("(\\.0)+$","");
    }

    /**
     * Method used to add effects to HapticMediaPlayer. Effects can be added from
     * internal storage and external storage. Both ways are provided for reference.
     */
    public void addEffects() {
        //NOTE: if you have only a few effects to add, then you can do this from the UI thread
        if (isHapticsInitialized()) {
            initStickerEffectsFromRawResources(); //load stickers from raw resources
        }

//        if (isHapticsInitialized()) {
//            initStickerEffectsFromExternalStorage(); //or you can load effects from external storage
//        }

        //Otherwise, it is recommended to add effects from a background thread
        //new AsyncHapticEffectAdder().execute();
    }

    /**
     * Method used to play haptic effects. If the effect was already added, its id would be
     * retrieved from the stickers list, and passed to HapticMediaPlayer instance for playback.
     *
     * @param effectName    The name of the effect added to HapticMediaPlayer
     * @param priority      The priority of the haptic effect,
     *                      one of HapticMediaPlayer.EffectPriority
     */
    public void playEffect(String effectName, int priority) {
        if (mErrorToast != null) {
            mErrorToast.show();
        }

        if (!isHapticsInitialized()) {
            return;
        }

        HapticEffect effect = getEffect(effectName);
        if (effect == null) {
            Log.e(TAG, "No effect added with name " + effectName);
            return;
        }

        int resourceId = effect.mResourceId;
        if (resourceId < 0) {
            Log.e(TAG, "Invalid resource id for effect with name " +
                    effectName + "! Error code returned = " + mErrorValues.get(resourceId));
            return;
        }

        stopAll();

        int effectId = mHapticMediaPlayer.play(resourceId, priority);
        if (effectId < 0) {
            Log.e(TAG, "Error while playing haptic effect with name " +
                    effectName + "! Error returned = " + mErrorValues.get(effectId));
        } else {
            effect.mEffectId = effectId;
        }
    }

    /**
     * Method used to pause haptic playback
     * @param effectName    The name of the effect added to HapticMediaPlayer
     */
    public void pauseEffect(String effectName) {
        if (!validateEffectIdAndHaptics(effectName)) {
            return;
        }

        int effectId = getEffectId(effectName);
        int retVal = mHapticMediaPlayer.pause(effectId);
        if (retVal != HapticMediaPlayer.TouchSenseSDKError.SUCCESS) {
            Log.e(TAG, "Error while pausing haptic effect with name " +
                    effectName + "! Error returned = " + mErrorValues.get(retVal));
        }
    }

    /**
     * Method used to resume haptic playback
     * @param effectName    The name of the effect added to HapticMediaPlayer
     */
    public void resumeEffect(String effectName) {
        if (!validateEffectIdAndHaptics(effectName)) {
            return;
        }

        int effectId = getEffectId(effectName);
        int retVal = mHapticMediaPlayer.resume(effectId);
        if (retVal != HapticMediaPlayer.TouchSenseSDKError.SUCCESS) {
            Log.e(TAG, "Error while resuming haptic effect with name " +
                    effectName + "! Error returned = " + mErrorValues.get(retVal));
        }
    }

    /**
     * Method used to resume haptic playback from a specified position
     * @param effectName    The name of the effect added to HapticMediaPlayer
     * @param positionMS    The desired position to resume haptic playback from, in milliseconds
     */
    public void seek(String effectName, int positionMS) {
        if (!validateEffectIdAndHaptics(effectName)) {
            return;
        }

        int effectId = getEffectId(effectName);
        int retVal = mHapticMediaPlayer.seek(effectId, positionMS);
        if (retVal != HapticMediaPlayer.TouchSenseSDKError.SUCCESS) {
            Log.e(TAG, "Error while seeking haptic effect with name " +
                    effectName + "! Error returned = " + mErrorValues.get(retVal));
        }
    }
    /**
     * Method used to update haptic playback from a specified position
     * @param effectName    The name of the effect added to HapticMediaPlayer
     * @param positionMS    The desired position to update haptic playback from, in milliseconds
     */
    public void update(String effectName, int positionMS) {
        if (!validateEffectIdAndHaptics(effectName)) {
            return;
        }

        if (!isEffectPlaying(getEffectId(effectName))) {
            return;
        }

        int effectId = getEffectId(effectName);
        int retVal = mHapticMediaPlayer.update(effectId, positionMS);
        if (retVal != HapticMediaPlayer.TouchSenseSDKError.SUCCESS) {
            Log.e(TAG, "Error while updating haptic effect with name " +
                    effectName + "! Error returned = " + mErrorValues.get(retVal));
        }
    }
    /**
     * Method used to stop haptic effects
     * @param effectName    The name of the effect added to HapticMediaPlayer
     */
    public void stopEffect(String effectName) {
        if (!validateEffectIdAndHaptics(effectName)) {
            return;
        }

        int effectId = getEffectId(effectName);
        int retVal = mHapticMediaPlayer.stop(effectId);
        if (retVal != HapticMediaPlayer.TouchSenseSDKError.SUCCESS) {
            Log.e(TAG, "Error while stopping haptic effect with name " +
                    effectName + "! Error returned = " + mErrorValues.get(retVal));
        }
    }

    /**
     * Method used to stop all haptic effects
     */
    public void stopAll() {
        if (!isHapticsInitialized()) {
            return;
        }
        for (HapticEffect effect : mStickerEffects) {
            int id = effect.mEffectId;
            if (id > 0) {
                stopEffect(effect.mEffectName);
                effect.mEffectId = 0;
            }
        }
    }

    /**
     * Method used to mute haptic effects
     * @param effectName    The name of the effect added to HapticMediaPlayer
     */
    public void muteEffect(String effectName) {
        if (!validateEffectIdAndHaptics(effectName)) {
            return;
        }

        int effectId = getEffectId(effectName);
        int retVal = mHapticMediaPlayer.mute(effectId);
        if (retVal != HapticMediaPlayer.TouchSenseSDKError.SUCCESS) {
            Log.e(TAG, "Error while muting haptic effect with name " +
                    effectName + "! Error returned = " + mErrorValues.get(retVal));
        }
    }

    /**
     * Method used to mute all haptic effects
     */
    public void muteAll() {
        if (!isHapticsInitialized()) {
            return;
        }
        for (HapticEffect effect : mStickerEffects) {
            int id = effect.mEffectId;
            if (id > 0 && isEffectPlaying(id)) {
                muteEffect(effect.mEffectName);
            }
        }
    }

    /**
     * Method used to unmute effects
     * @param effectName    The name of the effect added to HapticMediaPlayer
     */
    public void unmuteEffect(String effectName) {
        if (!validateEffectIdAndHaptics(effectName)) {
            return;
        }

        int effectId = getEffectId(effectName);
        int retVal = mHapticMediaPlayer.unmute(effectId);
        if (retVal != HapticMediaPlayer.TouchSenseSDKError.SUCCESS) {
            Log.e(TAG, "Error while unmuting haptic effect with name " +
                    effectName + "! Error returned = " + mErrorValues.get(retVal));
        }
    }

    /**
     * Method used to remove a haptic resource
     * @param effectName    The name of the effect added to HapticMediaPlayer
     */
    public void removeResource(String effectName) {
        if (!isHapticsInitialized()) {
            return;
        }

        HapticEffect effect = getEffect(effectName);
        if (effect == null) {
            Log.e(TAG, "No effect added with name " + effectName);
            return;
        }

        int resourceId = effect.mResourceId;
        if (resourceId < 0) {
            Log.e(TAG, "Invalid resource id for effect with name " +
                    effectName + "! Error code returned = " + mErrorValues.get(resourceId));
            return;
        }

        int retVal = mHapticMediaPlayer.removeResource(resourceId);
        if (retVal != HapticMediaPlayer.TouchSenseSDKError.SUCCESS) {
            Log.e(TAG, "Error while removing haptic resource with name " +
                    effectName + "! Error returned = " + mErrorValues.get(retVal));
        }
    }

    /**
     * Method used to dispose the HapticMediaPlayer object. Must be called in onDestroy
     * to make sure all resources held by HapticMediaPlayer are released.
     */
    public void dispose() {

        if (isHapticsInitialized()) {
            mHapticMediaPlayer.dispose();
        }
        if (mStickerEffects != null) {
            mStickerEffects.clear();
            mStickerEffects = null;
        }
        if (mErrorValues != null) {
            mErrorValues.clear();
            mErrorValues = null;
        }

        mHapticMediaPlayer = null;
        s_instance = null;
    }

    /**
     * Method that copies effects from res/raw/ directory to internal storage. Effects are then
     * added to the HapticMediaPlayer object using their full paths.
     */
    private void initStickerEffectsFromRawResources() {
        Field[] fields = R.raw.class.getFields();
        int rid;
        // loop for every file in raw folder
        for (int i = 0; i < fields.length; ++i) {
            try {
                if (fields[i].getType().getSimpleName().equals("IncrementalChange")) {
                    continue;
                }
                rid = fields[i].getInt(fields[i]);
            } catch (Exception e) {
                e.printStackTrace();
                continue;
            }
            String filename = fields[i].getName();
            String hapticPath = getHapticPath(rid, filename);

            //Add resource, retrieve id, and store it
            int resourceId = mHapticMediaPlayer.addResource(hapticPath, HapticMediaPlayer.HapticEffectType.ASYNC_HAPTIC_EFFECT);
            mStickerEffects.add(new HapticEffect(filename, resourceId));
        }
    }

    /**
     * Method that adds effects stored on external storage to the HapticMediaPlayer object. This method
     * assumes that EXTERNAL_STORAGE_DIRECTORY points to a valid directory with haptic effects
     */
    private void initStickerEffectsFromExternalStorage() {

        if (ContextCompat.checkSelfPermission(mActivity, Manifest.permission.READ_EXTERNAL_STORAGE) != PackageManager.PERMISSION_GRANTED) {
            //Request for permission to access external storage
            ActivityCompat.requestPermissions(mActivity,
                    new String[]{Manifest.permission.READ_EXTERNAL_STORAGE},
                    EXTERNAL_STORAGE_PERMISSION_REQUEST);
        } else {
            //Permission granted, load effects
            File effectsFilesDir = new File(getHaptDir());
            if (!effectsFilesDir.isDirectory()) {
                Log.e(TAG, "Directory with path " + EXTERNAL_STORAGE_DIRECTORY + " does not exist " +
                        "on external storage");
                return;
            }
            File[] files;
            int resourceId;
            files = effectsFilesDir.listFiles();
            for (File f : files) {
                if (f.getName().endsWith(HAPT_FILE_EXTENSION)) {
                    String effectName = f.getName().substring(0, f.getName().length() - 5);
                    resourceId = mHapticMediaPlayer.addResource(f.getAbsolutePath(),
                            HapticMediaPlayer.HapticEffectType.ASYNC_HAPTIC_EFFECT);
                    Log.d(TAG, "Adding effect " + effectName + " with id " + resourceId);
                    mStickerEffects.add(new HapticEffect(effectName, resourceId));
                }
            }
        }
    }

    private static String getHaptDir() {
        return Environment.getExternalStorageDirectory() + EXTERNAL_STORAGE_DIRECTORY;
    }

    private String getHapticPath(int id, String name) {
        FileOutputStream fos = null;
        int maxLen = 1024, curLen;
        byte[] tempBuffer = new byte[maxLen];
        InputStream is = mActivity.getResources().openRawResource(id);
        try {

            fos = mActivity.openFileOutput(name + HAPT_FILE_EXTENSION, Context.MODE_PRIVATE);
            curLen = is.available();
            while (curLen > 0) {
                curLen = is.read(tempBuffer);
                fos.write(tempBuffer, 0, curLen);
                curLen = is.available();
            }
            fos.close();
            is.close();
        } catch (FileNotFoundException fnfe) {
            try {
                is.close();
            } catch (Exception e) {
            }
        } catch (IOException ioe) {
            try {
                fos.close();
                is.close();
            } catch (Exception e) {
            }
        }

        // Retrieve the path of the hapt file now on disk
        File file = mActivity.getFilesDir();
        String path = file.getAbsolutePath() + File.separator + name + HAPT_FILE_EXTENSION;
        return path;
    }

    private void initHapticErrorValues() {
        mErrorValues = new HashMap<Integer, String>();

        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.SUCCESS, "SUCCESS");
        mErrorValues.put(HapticMediaPlayer.PlayerState.MISSING_PERMISSIONS, "MISSING_PERMISSIONS");
        mErrorValues.put(HapticMediaPlayer.PlayerState.BAD_PARAMETER, "INVALID_PARAMETER");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.INVALID_PARAMETER, "INVALID_PARAMETER");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.INVALID_URI, "INVALID_URI");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.INVALID_EFFECT, "INVALID_EFFECT");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.OUT_OF_MEMORY, "OUT_OF_MEMORY");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.IO_ERROR, "IO_ERROR");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.HAPT_NOT_READY, "HAPT_NOT_READY");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.TOO_MANY_EFFECTS, "TOO_MANY_EFFECTS");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.PLAYER_NOT_INITIALIZED, "PLAYER_NOT_INITIALIZED");
        mErrorValues.put(HapticMediaPlayer.PlayerState.INVALID_CREDENTIALS, "INVALID_CREDENTIALS");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.TOO_MANY_CONCURRENT_EFFECTS, "TOO_MANY_CONCURRENT_EFFECTS");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.INVALID_STATE, "INVALID_STATE");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.LIB_VERSION_NOT_FOUND, "LIB_VERSION_NOT_FOUND");
        mErrorValues.put(HapticMediaPlayer.TouchSenseSDKError.SDK_INOPERATIVE, "SDK_INOPERATIVE");
    }

    private HapticEffect getEffect(String effectName) {
        for (HapticEffect effect : mStickerEffects) {
            if (effect.mEffectName.equals(effectName)) {
                return effect;
            }
        }
        return null;
    }

    private int getEffectId(String effectName) {
        for (HapticEffect effect : mStickerEffects) {
            if (effect.mEffectName.equals(effectName)) {
                return effect.mEffectId;
            }
        }
        throw new IllegalArgumentException();
    }

    private boolean validateEffectIdAndHaptics(String effectName) {
        if (!isHapticsInitialized()) {
            return false;
        }

        try {
            int effectId = getEffectId(effectName);
            if (effectId < 0) {
                Log.e(TAG, "Invalid effect id for effect with name " +
                        effectName + "! Error code returned = " + effectId);
                return false;
            }
            return true;

        } catch (IllegalArgumentException e) {
            Log.e(TAG, "No effect added with name " + effectName);
            return false;
        }
    }

    /*
     * Async thread used to add effects from a background thread.
     * Recommended to be used when there are many effects to add,
     * in order to avoid blocking the UI thread.
     */
//    private class AsyncHapticEffectAdder extends AsyncTask<Void, Void, Integer> {
//
//        @Override
//        protected Integer doInBackground(Void... params) {
//
//            if (isHapticsInitialized()) {
//                initStickerEffectsFromRawResources();
//            }
//            return 0;
//        }
//
//        protected void onPostExecute() {
//            //Optional code to show result of adding effects
//        }
//    }
}
