/*
** =======================================================================
** Copyright (c) 2016  Immersion Corporation.  All rights reserved.
**                     Immersion Corporation Confidential and Proprietary.
**
** =======================================================================
*/

/*!
 * \file TouchSenseSDK.h
 * TouchSenseSDK entry point
 * This class represents a C enrty point interface into TouchSenseSDK. It
 * exposes all the APIs that allow third party Android applications to
 * integrate synchronous and asynchronous haptic effects
 */

#ifndef __TOUCHSENSE_SDK_H__
#define __TOUCHSENSE_SDK_H__

#define VISIBILITY_LVL __attribute__ ((visibility ("default")))

#ifdef __cplusplus
extern "C" {
#endif

/**
* @brief    Creates a HapticMediaPlayer instance
* @param    jvm       a pointer to JVM (set only when called from native)
* @param    context   a pointer to Java Context object
* @param    username  the user name provided by Immersion
* @param    password  the password provided by Immersion
* @param    dns       the dns url provided by Immersion
* @return   Pointer to the HapticMediaPlayer instance created
* @note     This function MUST be called first to retrieve a HapticMediaPlayer instance that you can work with
*/
VISIBILITY_LVL void* touchsensesdk_create(void* jvm, void** context, const char* username, const char* password, const char* dns);

/**
* @brief    Adds a haptic resource if it doesn't exists and returns its unique ID
* @param    player          Instance of the haptic media player returned by touchsensesdk_create
* @param    uri             URI to the effect .hapt file
* @param    playbackStyle   one of the values in {@link TSSDK_HapticEffectType}
* @return   if successful, a positive value - the resource's unique ID associated with the provided URI <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           {@link TSSDK_TOO_MANY_EFFECTS} if exceeded the maximum number of resources <br>
*           {@link TSSDK_OUT_OF_MEMORY} if there is insufficient memory to complete the operation <br>
*           {@link TSSDK_INVALID_PARAMETER} if the input is invalid
* @note     the resource ID returned by this function is to be used when calling removeEffect or play
*/
VISIBILITY_LVL int touchsensesdk_addResource(void* player, const char* uri, int playbackStyle);

/**
* @brief    Removes a haptic resource identified by its unique ID and stops any playing effects associated with it
* @param    player         Instance of the haptic media player returned by touchsensesdk_create
* @param    resourceId     unique ID of the resource to be removed
* @return   {@link TSSDK_SUCCESS} if the operation was successful <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           {@link TSSDK_INVALID_PARAMETER} if a resource with the provided ID is not found
*/
VISIBILITY_LVL int touchsensesdk_removeResource(void* player, int resourceId);

/**
* @brief    Plays a haptic resource identified by its unique ID
* @param    player        Instance of the haptic media player returned by touchsensesdk_create
* @param    resourceId    the ID of the {@link HapticResource} from which to create the effect
* @param    priority      one of the {@link TSSDK_EffectPriority} values indicating the effect's priority
* @return   if successful, a positive value - the effect's unique ID associated with the resource <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           {@link TSSDK_INVALID_PARAMETER} if a resource with the provided ID is not found or priority is invalid <br>
*           {@link TSSDK_INVALID_EFFECT} if the effect could not be created. See the log for more details <br>
*           {@link TSSDK_HAPT_NOT_READY} if the resource is not in state TSSDK_READY <br>
*           {@link TSSDK_TOO_MANY_CONCURRENT_EFFECTS} if maximum number of concurrent playbacks has been reached <br>
*           {@link TSSDK_OUT_OF_MEMORY} if there is insufficient memory to complete the operation
* @note     the effect ID returned by this function is to be used when calling pause, stop, update, seek, mute and unmute
* @note     the effect ID returned by this function is valid until the effect is stopped or completed
*/
VISIBILITY_LVL int touchsensesdk_play(void* player, int resourceId, int priority);

/**
* @brief    Pauses an effect identified by its unique ID
* @param    player       Instance of the haptic media player returned by touchsensesdk_create
* @param    effectId     unique ID of the effect to be paused
* @return   {@link TSSDK_SUCCESS} if the operation was successful <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           {@link TSSDK_INVALID_PARAMETER} if an effect with the provided ID is not found
*/
VISIBILITY_LVL int touchsensesdk_pause(void* player, int effectId);

/**
* @brief    Resumes an effect identified by its unique ID
* @param    player       Instance of the haptic media player returned by touchsensesdk_create
* @param    effectId     unique ID of the effect to be resumed
* @return   {@link TSSDK_SUCCESS} if the operation was successful <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           {@link TSSDK_INVALID_PARAMETER} if an effect with the provided ID is not found
*/
VISIBILITY_LVL int touchsensesdk_resume(void* player, int effectId);

/**
* @brief    Stops an effect identified by its unique ID
* @param    player       Instance of the haptic media player returned by touchsensesdk_create
* @param    effectId     unique ID of the effect to be stopped
* @return   {@link TSSDK_SUCCESS} if the operation was successful <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           {@link TSSDK_INVALID_PARAMETER} if an effect with the provided ID is not found
*/
VISIBILITY_LVL int touchsensesdk_stop(void* player, int effectId);

/**
* @brief    Updates an effect identified by its unique ID to a specific timestamp
* @param    player           Instance of the haptic media player returned by touchsensesdk_create
* @param    effectId         unique ID of the effect to be updated
* @param    timestampMS timestamp (in milliseconds) to update to
* @return   {@link TSSDK_SUCCESS} if the operation was successful <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           {@link TSSDK_INVALID_PARAMETER} if an effect with the provided ID is not found
*/
VISIBILITY_LVL int touchsensesdk_update(void* player, int effectId, long timestampMS);

/**
* @brief    Seeks an effect identified by its unique ID to a specific timestamp
* @param    player           Instance of the haptic media player returned by touchsensesdk_create
* @param    effectId         unique ID of the effect to be updated
* @param    timestampMS timestamp (in milliseconds) to seek to
* @return   {@link TSSDK_SUCCESS} if the operation was successful <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           {@link TSSDK_INVALID_PARAMETER} if an effect with the provided ID is not found
*/
VISIBILITY_LVL int touchsensesdk_seek(void* player, int effectId, long timestampMS);

/**
* @brief    Mutes an effect identified by its unique ID
* @param    player       Instance of the haptic media player returned by touchsensesdk_create
* @param    effectId     unique ID of the effect to be muted
* @return   {@link TSSDK_SUCCESS} if the operation was successful <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           {@link TSSDK_INVALID_PARAMETER} if an effect with the provided ID is not found
*/
VISIBILITY_LVL int touchsensesdk_mute(void* player, int effectId);

/**
* @brief    Un-mutes an effect identified by its unique ID
* @param    player       Instance of the haptic media player returned by touchsensesdk_create
* @param    effectId     unique ID of the effect to be un-muted
* @return   {@link TSSDK_SUCCESS} if the operation was successful <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           {@link TSSDK_INVALID_PARAMETER} if an effect with the provided ID is not found
*/
VISIBILITY_LVL int touchsensesdk_unmute(void* player, int effectId);

/**
* @brief    Retrieves information on the HapticMediaPlayer according to information code provided
* @param    player       Instance of the haptic media player returned by touchsensesdk_create
* @param    info         one of the values in {@link TSSDK_PlayerInfo}
* @return   {@link TSSDK_INVALID_PARAMETER} if info is not a valid value <br>
*           for {@link TSSDK_PLAYER_STATE}               - one of the values in {@link TSSDK_PlayerState} <br>
*           for {@link TSSDK_PLAYER_VERSION_MAJOR}       - Major version of the HapticMediaPlayer <br>
*           for {@link TSSDK_PLAYER_VERSION_MINOR}       - Minor version of the HapticMediaPlayer <br>
*           for {@link TSSDK_PLAYER_VERSION_BUILD}       - Build version of the HapticMediaPlayer <br>
*           for {@link TSSDK_PLAYER_VERSION_MAINTENANCE} - Maintenance version of the HapticMediaPlayer <br>
*           for {@link TSSDK_PLAYER_VERSION_PATCH}       - Patch version number of the HapticMediaPlayer <br>
*           if any of the TSSDK_PLAYER_VERSION_* values is missing {@link TSSDK_LIB_VERSION_NOT_FOUND} <br>
*           for {@link TSSDK_PLAYER_PLAYBACK_TYPE}       - one of the values in {@link TSSDK_PlaybackType} <br>
*                                                          or {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized
*/
VISIBILITY_LVL int touchsensesdk_getPlayerInfo(void* player, int info);

/**
* @brief    Retrieves information on an effect according to its unique ID and information code provided
* @param    player       Instance of the haptic media player returned by touchsensesdk_create
* @param    uid          the unique ID of the resource or effect to retrieve information for
* @param    info         one of the values in {@link TSSDK_EffectInfo}
* @return   {@link TSSDK_INVALID_PARAMETER} if an effect with the provided uid is not found or info not a valid value <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized <br>
*           for {@link TSSDK_RESOURCE_STATE} - one of the values in {@link TSSDK_ResourceState} <br>
*           for {@link TSSDK_EFFECT_STATE} - one of the values in {@link TSSDK_EffectState} <br>
*           for {@link TSSDK_EFFECT_MUTED_STATE} - one of the values in {@link TSSDK_EffectMutedState} <br>
*           for {@link TSSDK_RESOURCE_TYPE} - one of the values in {@link TSSDK_HapticEffectType} <br>
*           for {@link TSSDK_RESOURCE_LOCATION} - one of the values in {@link TSSDK_HapticEffectLocation} <br>
*           for {@link TSSDK_RESOURCE_DURATION} - duration in milliseconds, or {@link TSSDK_HAPT_NOT_READY} if the resource is no ready <br>
* @note     resource uid is retrieved when calling addResource
* @note     effect uid is retrieved when calling play
*/
VISIBILITY_LVL int touchsensesdk_getEffectInfo(void* player, int uid, int info);

/**
* @brief    Disposes of the HapticMediaPlayer instance, all existing resources and any playing effects
* @param    player       Instance of the haptic media player returned by touchsensesdk_create
* @return   {@link TSSDK_SUCCESS} if operation was successful <br>
*           {@link TSSDK_PLAYER_NOT_INITIALIZED} if the SDK is not initialized
* @note     Following this call the player instance will no longer be valid
*/
VISIBILITY_LVL int touchsensesdk_dispose(void* player);

#ifdef __cplusplus
}
#endif

#endif // __TOUCHSENSE_SDK_H__
