<?php
/**
 * File: class-wpglobus-header-images.php
 *
 * @package WPGlobus_Header_Images
 */

if ( ! class_exists( 'WPGlobus_Header_Images' ) ) :

	/**
	 * Class WPGlobus_Header_Images
	 */
	class WPGlobus_Header_Images {

		/**
		 * Plugin options
		 *
		 * @var array
		 */
		static protected $options = array();

		/**
		 * Array of attachments that is custom headers.
		 *
		 * @var array
		 */
		static protected $ids = array();

		/**
		 * The options key.
		 *
		 * @var string
		 */
		static public $option_key = '_wpglobus_header_images';

		/**
		 * The redirect key.
		 *
		 * @var string
		 */
		static public $redirect_key = 'wpglobus_header_images';

		/**
		 * Current theme.
		 *
		 * @var WP_Theme
		 */
		public static $theme;

		/**
		 * Keeps the `__FILE__` of the plugin's loader.
		 *
		 * @var string
		 */
		protected static $PLUGIN_FILE;

		/**
		 * The filesystem path of the directory that contains the plugin.
		 *
		 * @var string
		 */
		protected static $PLUGIN_DIR_PATH;

		/**
		 * The URL path of the directory that contains the plugin.
		 *
		 * @var string
		 */
		protected static $PLUGIN_DIR_URL;

		/**
		 * The name of a plugin extracted from its filename.
		 *
		 * @var string
		 */
		protected static $PLUGIN_BASENAME;

		/**
		 * Static constructor.
		 *
		 * @param string $path_to_loader Plugin loader must pass its `__FILE__`.
		 */
		public static function construct( $path_to_loader ) {

			self::$PLUGIN_FILE     = $path_to_loader;
			self::$PLUGIN_DIR_PATH = plugin_dir_path( self::$PLUGIN_FILE );
			self::$PLUGIN_DIR_URL  = plugin_dir_url( self::$PLUGIN_FILE );
			self::$PLUGIN_BASENAME = plugin_basename( self::$PLUGIN_FILE );

			self::$theme = wp_get_theme();
			self::$option_key = self::$option_key . '_' . get_stylesheet();
			
			self::$options    = get_option( self::$option_key );
			if ( empty( self::$options ) ) {
				self::$options = array();
			}

			/**
			 * AJAX handlers.
			 *
			 * @scope admin
			 */
			if ( version_compare( WPGLOBUS_HEADER_IMAGES_VERSION, '2.0.0', '>=' ) ) {
				add_action( 'wp_ajax_' . __CLASS__ . '_process_ajax', array( __CLASS__, 'ajax_events_2' ) );
			} else {
				add_action( 'wp_ajax_' . __CLASS__ . '_process_ajax', array( __CLASS__, 'ajax_events' ) );
			}

			if ( is_admin() ) {

				/**
				 * Enqueue admin scripts & styles.
				 *
				 * @scope admin
				 */
				if ( version_compare( WPGLOBUS_HEADER_IMAGES_VERSION, '2.0.0', '>=' ) ) {

					// phpcs:ignore WordPress.CSRF.NonceVerification
					if ( ! empty( $_GET['page'] ) && $_GET['page'] === self::$redirect_key ) {

						$url = admin_url(
							add_query_arg(
								array(
									'wpglobus-header-images' => '',
								),
								'upload.php?mode=list'
							)
						);
						wp_redirect( $url );
						exit;

					}

					add_action( 'admin_menu', array( __CLASS__, 'on__admin_menu' ) );
					add_action( 'admin_print_scripts', array( __CLASS__, 'admin_print_scripts_2' ) );
					add_action( 'admin_print_styles', array( __CLASS__, 'admin_print_styles' ) );
					add_action( 'pre_get_posts', array( __CLASS__, 'on__pre_get_posts' ) );
					add_action( 'admin_footer', array( __CLASS__, 'on__admin_footer' ), 9999 );

				} else {
					add_action( 'admin_print_scripts', array( __CLASS__, 'admin_print_scripts' ) );
					add_action( 'admin_print_styles', array( __CLASS__, 'admin_print_styles' ) );
				}

				/**
				 * Add an action link.
				 *
				 * @scope admin
				 */
				add_filter( 'plugin_action_links_' . self::$PLUGIN_BASENAME,
					array( __CLASS__, 'filter__plugin_action_links' )
				);

				/**
				 * @since 2.1.0
				 */
				add_filter( 'wpglobus_option_sections', array( __CLASS__, 'filter__add_section' ) );

			} else {

				/**
				 * Filter for theme_mods_ option.
				 *
				 * @scope front
				 */
				add_filter( 'option_theme_mods_' . get_stylesheet(), array(
					__CLASS__,
					'filter__option',
				), 5 );
			}
		}

		/**
		 * Callback to filter for theme_mods_ option.
		 *
		 * @since 1.0.0
		 * @scope front
		 *
		 * @param array $opts
		 *
		 * @return mixed
		 */
		public static function filter__option( $opts ) {

			if ( WPGlobus::Config()->language === WPGlobus::Config()->default_language ) {
				return $opts;
			}
			if ( empty( self::$options[ WPGlobus::Config()->language ] ) ) {
				return $opts;
			}

			if ( in_array( $opts['header_image'], array( 'remove-header', 'random-uploaded-image' ), true ) ) {
				/**
				 * Don't change image for extra language if Current header for default language was removed.
				 */
				return $opts;
			}

			if ( ! empty( $opts['header_image'] ) ) {
				$opts['header_image'] = self::$options[ WPGlobus::Config()->language ]['img_data']['url'];
			}
			if ( ! empty( $opts['header_image_data'] ) ) {
				$opts['header_image_data']->attachment_id = self::$options[ WPGlobus::Config()->language ]['img_data']['attachment_id'];
				$opts['header_image_data']->url           = self::$options[ WPGlobus::Config()->language ]['img_data']['url'];
				$opts['header_image_data']->thumbnail_url = self::$options[ WPGlobus::Config()->language ]['img_data']['thumbnail_url'];
				$opts['header_image_data']->height        = self::$options[ WPGlobus::Config()->language ]['img_data']['height'];
				$opts['header_image_data']->width         = self::$options[ WPGlobus::Config()->language ]['img_data']['width'];
			}

			return $opts;
		}

		/**
		 * Callback for `pre_get_posts` action.
		 *
		 * @since 2.0.0
		 *
		 * @param $query
		 *
		 * @return void
		 */
		public static function on__pre_get_posts( $query ) {

			global $wpdb;

			$sql = $wpdb->prepare( "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value = %s", '_wp_attachment_is_custom_header', get_stylesheet() );

			/** @var array $ids */
			// phpcs:ignore WordPress.WP.PreparedSQL
			$ids = $wpdb->get_results( $sql, ARRAY_A );

			if ( ! empty( $ids ) ) {
				self::$ids = wp_list_pluck( $ids, 'post_id' );
			}

			// phpcs:ignore WordPress.CSRF.NonceVerification
			if ( isset( $_GET['wpglobus-header-images'] ) && ! empty( self::$ids ) ) {
				$query->query_vars['post__in'] = self::$ids;
			}

		}

		/**
		 * Add submenu.
		 *
		 * @since 2.0.0
		 * @return void
		 */
		public static function on__admin_menu() {

			add_submenu_page(
				'upload.php',
				'Header images',
				'Header images',
				'administrator',
				self::$redirect_key,
				array(
					__CLASS__,
					'on__header_images',
				)
			);

		}

		/**
		 * @since 2.0.0
		 * @return void
		 */
		public static function on__header_images() {
			// do nothing.
		}

		/**
		 * AJAX events.
		 *
		 * @since 2.0.0
		 * @return void
		 */
		public static function ajax_events_2() {

			$order = array();

			// phpcs:ignore WordPress.CSRF.NonceVerification
			if ( isset( $_POST ) && ! empty( $_POST['order'] ) ) {
				// phpcs:ignore WordPress.CSRF.NonceVerification
				$order = wp_unslash( $_POST['order'] );
			}

			$result = true;

			if ( isset( $order['action'] ) ) :
				switch ( $order['action'] ) :
					case 'get_options_!!!':
						$order['options'] = self::$options;
						break;
					case 'delete_!!!':
						$result = delete_option( self::$option_key );
						break;
					case 'delete_option':
						$update_option = false;

						$opts = get_option( self::$option_key );

						foreach ( $opts as $language => $opt ) {
							if ( isset( $order['language'] ) && $language === $order['language'] ) {
								unset( $opts[ $language ] );
								$update_option = true;
								break;
							}
						}

						if ( $update_option ) {
							if ( empty( $opts ) ) {
								delete_option( self::$option_key );
							} else {
								$result = update_option( self::$option_key, $opts, false );
							}
						}

						break;
					case 'save':
						$img_data = $order['img_data'];

						if ( empty( $img_data ) ) {
							$result = false;
						} else {

							// $attachment_id = $img_data['attachment_id'];

							// phpcs:ignore Generic.CodeAnalysis.EmptyStatement
							if ( empty( $order['language'] ) ) {
								// incorrect case.
							} else {

								self::$options[ $order['language'] ]['version']  = WPGLOBUS_HEADER_IMAGES_VERSION;
								self::$options[ $order['language'] ]['img_data'] = $img_data;
								self::$options[ $order['language'] ]['sticker']  = $order['sticker'];
								if ( empty( $order['sticker_data'] ) ) {
									self::$options[ $order['language'] ]['sticker_data'] = array();
								} else {
									self::$options[ $order['language'] ]['sticker_data'] = $order['sticker_data'];
								}
							}

							if ( empty( self::$options ) ) {
								delete_option( self::$option_key );
							} else {
								$result = update_option( self::$option_key, self::$options, false );
							}
						}

						break;

					default:
				endswitch;
			endif;

			if ( false === $result ) {
				wp_send_json_error( $order );
			}

			wp_send_json_success( $order );

		}

		/**
		 * Ajax events handler.
		 *
		 * @since 1.0.0
		 * @scope admin
		 */
		public static function ajax_events() {

			$order = array();

			// phpcs:ignore WordPress.CSRF.NonceVerification
			if ( isset( $_POST ) && ! empty( $_POST['order'] ) ) {
				// phpcs:ignore WordPress.CSRF.NonceVerification
				$order = wp_unslash( $_POST['order'] );
			}

			$result = true;

			if ( isset( $order['action'] ) ) :
				switch ( $order['action'] ) :
					case 'get_options':
						$order['options'] = self::$options;
						break;
					case 'delete':
						$result = delete_option( self::$option_key );
						break;
					case 'delete_option':
						$update_option = false;

						$opts = get_option( self::$option_key );

						foreach ( $opts as $lang => $opt ) {
							if ( false !== strpos( $order['data'], $opt['img_data']['attachment_id'] ) ) {
								unset( $opts[ $lang ] );
								$update_option = true;
								break;
							}
						}

						if ( $update_option ) {
							if ( empty( $opts ) ) {
								delete_option( self::$option_key );
							} else {
								$result = update_option( self::$option_key, $opts, false );
							}
						}

						break;
					case 'save':
						$img_data = $order['img_data'];

						if ( empty( $img_data ) ) {
							$result = false;
						} else {

							$attachment_id = $img_data['attachment_id'];

							if ( empty( $order['language'] ) ) {
								/**
								 * sticker 'empty'
								 */
								foreach ( self::$options as $lang => $data ) {
									if ( $data['img_data']['attachment_id'] === $attachment_id ) {
										unset( self::$options[ $lang ] );
									}
								}
							} else {
								if ( ! empty( self::$options ) ) {
									foreach ( self::$options as $lang => $data ) {
										if ( $data['img_data']['attachment_id'] === $attachment_id ) {
											unset( self::$options[ $lang ] );
										}
									}
								}

								/**
								 * Don't rewrite existing setting of another image
								 */
								if ( empty( self::$options[ $order['language'] ] ) ) {
									self::$options[ $order['language'] ]['img_data']     = $img_data;
									self::$options[ $order['language'] ]['sticker']      = $order['sticker'];
									self::$options[ $order['language'] ]['sticker_data'] = $order['sticker_data'];
								}
							}

							if ( empty( self::$options ) ) {
								delete_option( self::$option_key );
							} else {
								$result = update_option( self::$option_key, self::$options, false );
							}
						}

						break;
					default:
				endswitch;
			endif;

			if ( false === $result ) {
				wp_send_json_error( $order );
			}

			wp_send_json_success( $order );

		}

		/**
		 * Add a link.
		 *
		 * @since 2.1.0
		 * @scope admin
		 *
		 * @param array $links An array of plugin action links.
		 *
		 * @return array $links Array with the added link.
		 */
		public static function filter__plugin_action_links( $links ) {
			$_url = admin_url(
				add_query_arg(
					array( 'page' => 'wpglobus_options', 'tab' => 'wpglobus-header-images' ),
					'admin.php'
				)
			);
			$settings_link = '<a class="" href="'.$_url.'">' . esc_html__( 'Info', 'wpglobus-header-images' ) . '</a>';
			array_unshift( $links, $settings_link );
			return $links;
		}

		/**
		 * Add option section on WPGlobus Option page.
		 * 
		 * @since 2.1.0
		 */
		public static function filter__add_section( $sections ) {

			$_url = admin_url(
				add_query_arg(
					array('section'=>'header_image'),
					'customize.php'
				)
			);

			$_row1 = sprintf( esc_html__( 'Open the `Header image` section in %1$sCustomizer%2$s', 'wpglobus-header-images' ), '<a href="' . $_url . '">', '</a>' );
			$_row2 = esc_html__( 'Add the images', 'wpglobus-header-images' );
			$_row3 = esc_html__( 'Press the `Publish` button', 'wpglobus-header-images' );
			$_row4 = esc_html__( 'Press the `Set WPGlobus Header Images` button', 'wpglobus-header-images' );
			$_row5 = esc_html__( 'On the `Header image` admin page, assign image to each language', 'wpglobus-header-images' );
			$_row6 = esc_html__( 'Verify the results at the frontend', 'wpglobus-header-images' );

			$_info_desc  = '<ol>';
			$_info_desc .= '<li style="font-size:18px;">' . $_row1 . '</li>';
			$_info_desc .= '<li style="font-size:18px;">' . $_row2 . '</li>';
			$_info_desc .= '<li style="font-size:18px;">' . $_row3 . '</li>';
			$_info_desc .= '<li style="font-size:18px;">' . $_row4 . '</li>';
			$_info_desc .= '<li style="font-size:18px;">' . $_row5 . '</li>';
			$_info_desc .= '<li style="font-size:18px;">' . $_row6 . '</li>';
			$_info_desc .= '</ol>';

			$fields[] = array(
				'id'    => 'wpglobus_header_images_info',
				'type'  => 'wpglobus_info',
				'title' => esc_html__( 'How to set header images varying by languages', 'wpglobus-header-images' ),
				'desc'  => $_info_desc,
				'class' => 'info', // or normal
			);
		
			$sections['wpglobus-header-images'] = array(
				'wpglobus_id' => 'wpglobus_header_images',
				'title'       => esc_html__( 'Header Images', 'wpglobus-header-images' ),
				'caption'     => esc_html__( 'WPGlobus Header Images', 'wpglobus-header-images' ),
				'icon'        => 'dashicons dashicons-format-gallery',
				'fields'      => $fields,
			);
			
			return $sections;
		}
		
		public static function admin_print_scripts_2() {

			global $pagenow;

			if ( WPGlobus_WP::is_pagenow( array( 'upload.php' ) ) ) {

				$select = 'for <select id="wpglobus-header-images-select-{{post-id}}" class="wpglobus-header-images-select" data-id="{{post-id}}" data-language="{{language}}" data-customize-header-image-data="">';

				$default_select = $select;

				$default_select .= '<option>' . WPGlobus::Config()->en_language_name[ WPGlobus::Config()->default_language ] . '</option>';
				$default_select .= '</select>';

				$select .= '<option value="reset-select">---</option>';
				foreach ( WPGlobus::Config()->enabled_languages as $language ) {
					if ( WPGlobus::Config()->default_language !== $language ) {
						$select .= '<option {{selected-' . $language . '}} value="' . $language . '">' . WPGlobus::Config()->en_language_name[ $language ] . '</option>';
					}
				}
				$select .= '</select>';

				wp_register_script(
					'wpglobus-header-images',
					plugin_dir_url( __FILE__ ) . 'js/wpglobus-header-images-2' . WPGlobus::SCRIPT_SUFFIX() . '.js',
					array( 'jquery' ),
					WPGLOBUS_HEADER_IMAGES_VERSION,
					true
				);
				wp_enqueue_script( 'wpglobus-header-images' );

				$default_header_image    = get_header_image();
				$default_header_image_id = 0;

				$images = array();
				foreach ( self::$ids as $id ) {

					$header_image = wp_get_attachment_url( $id );

					if ( $header_image === $default_header_image ) {
						$default_header_image_id   = $id;
						$images[ $id ]['language'] = WPGlobus::Config()->default_language;
					} else {
						$images[ $id ]['language'] = '';
					}

					$metadata = wp_get_attachment_metadata( $id );

					$images[ $id ]['url']           = $header_image;
					$images[ $id ]['thumbnail_url'] = $header_image;
					$images[ $id ]['width']         = $metadata['width'];
					$images[ $id ]['height']        = $metadata['height'];

				}

				foreach ( self::$options as $language => $data ) {
					// phpcs:ignore Generic.CodeAnalysis.EmptyStatement
					if ( $data['img_data']['attachment_id'] === $default_header_image_id ) {
						// if image ID in options is the same as image ID for default language, then do nothing.
					} else {
						$images[ $data['img_data']['attachment_id'] ]['language'] = $language;
					}
				}

				wp_localize_script(
					'wpglobus-header-images',
					'WPGlobusHeaderImages',
					array(
						'version'              => WPGLOBUS_HEADER_IMAGES_VERSION,
						'pagenow'              => $pagenow,
						'dummyLink'            => 'upload.php?page=' . self::$redirect_key,
						'link'                 => 'upload.php?mode=list&wpglobus-header-images',
						'images'               => $images,
						'defaultHeaderImageId' => $default_header_image_id,
						'defaultLanguage'      => WPGlobus::Config()->default_language,
						'enabledLanguages'     => WPGlobus::Config()->enabled_languages,
						'enLanguageName'       => WPGlobus::Config()->en_language_name,
						'select'               => $select,
						'defaultSelect'        => $default_select,
						'ajaxurl'              => admin_url( 'admin-ajax.php' ),
						'process_ajax'         => __CLASS__ . '_process_ajax',
					)
				);

			} elseif ( WPGlobus_WP::is_pagenow( array( 'customize.php' ) ) ) {

				wp_register_script(
					'wpglobus-header-images',
					plugin_dir_url( __FILE__ ) . 'js/wpglobus-header-images-2' . WPGlobus::SCRIPT_SUFFIX() . '.js',
					array( 'jquery' ),
					WPGLOBUS_HEADER_IMAGES_VERSION,
					true
				);
				wp_enqueue_script( 'wpglobus-header-images' );

				$default_header_image    = get_header_image();
				$default_header_image_id = 0;

				$ids = array();
				foreach ( self::$ids as $id ) {
					$header_image = wp_get_attachment_url( $id );
					if ( $header_image === $default_header_image ) {
						$default_header_image_id = $id;
						$ids[ $id ]              = WPGlobus::Config()->default_language;
						break;
					}
				}

				foreach ( self::$options as $language => $data ) {
					// phpcs:ignore Generic.CodeAnalysis.EmptyStatement
					if ( $data['img_data']['attachment_id'] === $default_header_image_id ) {
						// if image ID in options is the same as image ID for default language, then do nothing.
					} else {
						$ids[ $data['img_data']['attachment_id'] ] = $language;
					}
				}

				$default_sticker = '<span style="float: left;font-size: 13px;font-weight: 400;background-color: #0085ba;padding: 15px 15px;color: #fff;margin-top: -2px;">{{language}}</span>';
				$sticker         = '<span style="float: left;font-size: 13px;font-weight: 400;background-color: #0085ba;padding: 5px 10px;border-top-left-radius: 10px;color: #fff;margin-top: -2px;">{{language}}</span>';

				$link = admin_url(
					add_query_arg(
						array(
							'wpglobus-header-images' => '',
						),
						'upload.php?mode=list'
					)
				);

				wp_localize_script(
					'wpglobus-header-images',
					'WPGlobusHeaderImages',
					array(
						'version'              => WPGLOBUS_HEADER_IMAGES_VERSION,
						'pagenow'              => $pagenow,
						'link'                 => $link,
						'ids'                  => $ids,
						'sticker'              => $sticker,
						'defaultSticker'       => $default_sticker,
						'defaultHeaderImageId' => $default_header_image_id,
						'defaultLanguage'      => WPGlobus::Config()->default_language,
						'enabledLanguages'     => WPGlobus::Config()->enabled_languages,
						'enLanguageName'       => WPGlobus::Config()->en_language_name,
						'options'              => self::$options,
						'get'				   => $_GET
					)
				);

			}

		}

		/**
		 * obsolete.
		 *
		 * Enqueue admin scripts.
		 *
		 * @since 1.0.0
		 * @scope admin
		 *
		 * @return void
		 */
		public static function admin_print_scripts() {

			if ( ! WPGlobus_WP::is_pagenow( array( 'customize.php' ) ) ) {
				return;
			}

			$i18n = array();

			$opt = get_option( 'theme_mods_' . get_stylesheet() );

			$theme_options = null;

			if ( ! empty( $opt['header_image'] ) ) {
				$theme_options['header_image'] = $opt['header_image'];
			}
			if ( ! empty( $opt['header_image_data'] ) ) {
				$theme_options['header_image_data'] = $opt['header_image_data'];
			}

			// phpcs:ignore WordPress.PHP.StrictComparisons
			if ( null == $theme_options || empty( $theme_options['header_image'] ) || empty( $theme_options['header_image'] ) ) {
				$i18n['saveOptions']['selector'] = '#customize-control-header_image .customizer-section-intro';
				$i18n['saveOptions']['message']  = __( 'To start using the WPGlobus Header Images plugin, you need to add an image to the <strong>Current header</strong> and then click Save&amp;Publish and reload the page.', 'wpglobus-header-images' );
			}

			wp_register_script(
				'wpglobus-header-images',
				plugin_dir_url( __FILE__ ) . 'js/wpglobus-header-images' . WPGlobus::SCRIPT_SUFFIX() . '.js',
				array( 'jquery' ),
				WPGLOBUS_HEADER_IMAGES_VERSION,
				true
			);
			wp_enqueue_script( 'wpglobus-header-images' );

			wp_localize_script(
				'wpglobus-header-images',
				'WPGlobusHeaderImages',
				array(
					'version'              => WPGLOBUS_HEADER_IMAGES_VERSION,
					'imagesSelector'       => '#customize-control-header_image .header-view',
					'imageCurrentSelector' => '#customize-control-header_image .current .container',
					'stickerSelector'      => '#customize-control-header_image .language-sticker',
					'themeOptions'         => $theme_options,
					'options'              => null, // load via js ajax
					'process_ajax'         => __CLASS__ . '_process_ajax',
					'i18n'                 => $i18n,
					#'_wp_attachment_is_custom_header' => '_wp_attachment_is_custom_header' // @see DB table postmeta
				)
			);
		}

		/**
		 * Enqueue admin styles
		 *
		 * @since 1.0.0
		 * @scope admin
		 *
		 * @return void
		 */
		public static function admin_print_styles() {

			if ( ! WPGlobus_WP::is_pagenow( array( 'customize.php' ) ) ) {
				return;
			}

			wp_register_style(
				'wpglobus-header-images',
				plugin_dir_url( __FILE__ ) . '/css/wpglobus-header-images.css',
				array(),
				WPGLOBUS_HEADER_IMAGES_VERSION,
				'all'
			);
			wp_enqueue_style( 'wpglobus-header-images' );

		}

		/**
		 * Debug.
		 *
		 * @scope admin
		 * @since 2.0.0
		 */
		public static function on__admin_footer() {
			// phpcs:ignore WordPress.CSRF.NonceVerification
			if ( ! isset( $_GET['wpglobus-header-images'] ) ) {
				return;
			}
			// phpcs:ignore WordPress.CSRF.NonceVerification
			if ( ! isset( $_GET['wpglobus-debug'] ) ) {
				return;
			}
			?>
			<div id="wpglobus-admin-debug-box" style="display:block;">
				<h4>WPGlobus debug box</h4>
				<pre><?php echo print_r( self::$options, true ); // phpcs:ignore ?><br/><br/></pre>
			</div>
			<?php
		}

	}
endif;
